package ru.yandex.mail.search.web.drop;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.PropertyStatus;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.parser.searchmap.SearchMap;

public class DropLuceneShardHandler
    extends AbstractDropHandler
    implements ProxyRequestHandler
{
    private final HealthCheckService healthService;

    public DropLuceneShardHandler(final HealthCheckService healthService) {
        this.healthService = healthService;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        String host = session.params().getString("host");
        Integer shardId = session.params().getInt("shard");
        int luceneShards = healthService.project().config().luceneShards();
        Integer luceneShardId = shardId % luceneShards;

        List<ShardDroppableCheckResult> shards = new ArrayList<>();
        session.logger().info("Lucene Shard is " + luceneShardId);

        Set<String> queues = new LinkedHashSet<>();
        Set<Integer> basePorts = new LinkedHashSet<>();
        for (ProjectQueue queue
            : healthService.projectRoot().queues().values())
        {
            for (int i = luceneShardId;
                 i < SearchMap.SHARDS_COUNT; i += luceneShards)
            {
                Shard shard = queue.shard(i);
                for (ShardReplica directReplica: shard.replicas().values()) {
                    if (host.equalsIgnoreCase(directReplica.host().hostname())) {
                        basePorts.add(directReplica.host().basePort());
                        shards.add(calcShard(shard, directReplica));
                        queues.add(queue.service());
                        session.logger().info(
                            "Found shard " + shard.queue().service()
                                + " " + i + " index port "
                                + directReplica.host().indexerPort());
                    }
                }
            }
        }

        PropertyStatus worstStatus = PropertyStatus.OK;

        for (ShardDroppableCheckResult result: shards) {
            if (result.status().ordinal() > worstStatus.ordinal()) {
                worstStatus = result.status();
            }
        }

        boolean drop = session.params().getBoolean("drop", false);
        if (drop) {
            if (basePorts.size() > 1) {
                throw new BadRequestException(
                    "Multi containers on one host are not supported by now "
                        + basePorts);
            }

            if (worstStatus == PropertyStatus.OK
                || worstStatus == PropertyStatus.WARNING)
            {
                String dropPolicy = session.params().getString("drop_policy", null);
                DropShardContext contenxt = new DropShardContext(
                    host,
                    basePorts.iterator().next(),
                    shards,
                    luceneShardId,
                    healthService.project(),
                    session,
                    queues,
                    dropPolicy);
                dropShard(contenxt);
                return;
            } else {
                session.logger().warning("Bad status");
            }
        }

        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
            writer.startObject();
            writer.key("status");
            writer.value(worstStatus.name());
            writer.key("lucene_shard");
            writer.value(luceneShardId);
            writer.key("shards");
            writer.startArray();
            for (ShardDroppableCheckResult result: shards) {
                writer.value(result);
            }
            writer.endArray();
            writer.endObject();
        }

        session.response(HttpStatus.SC_OK, new StringEntity(
            sbw.toString(),
            StandardCharsets.UTF_8));
    }

    protected void dropShard(
        final DropShardContext contenxt)
    {
        contenxt.logger().warning(
            "Dropping shard " + contenxt.luceneShard()
                + " on " + contenxt.hostname());
        StringBuilder uri = new StringBuilder("/shard-drop-and-copy?shard=");
        uri.append(contenxt.luceneShard());
        if (contenxt.dropPolicy() != null) {
            uri.append("&drop_policy=");
            uri.append(contenxt.dropPolicy());
        }
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(uri.toString());

        generator.addHeader(
            "ticket",
            contenxt.project().config().luceneDropPassword());

        contenxt.client().execute(
            contenxt.indexerHost(),
            generator,
            EmptyAsyncConsumerFactory.ANY_GOOD,
            contenxt.session().listener().createContextGeneratorFor(
                contenxt.client()),
            new ActualDropShardCallback(contenxt));
    }

    private static final class ResetShardCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final DropShardContext context;
        private final boolean error;
        private final String message;

        public ResetShardCallback(
            final DropShardContext context,
            final boolean error,
            final String message)
        {
            super(context.session());
            this.context = context;
            this.error = error;
            this.message = message;
        }

        @Override
        public void completed(final Object result) {
            session.logger().warning("Dropped");
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
                writer.startObject();
                writer.key("status");
                writer.value(PropertyStatus.OK);
                writer.key("error");
                writer.value(error);
                writer.key("message");
                writer.value(message);
                writer.key("lucene_shard");
                writer.value(context.luceneShard());
                writer.key("host");
                writer.value(context.hostname());
                writer.endObject();
            } catch (IOException ioe) {
                failed(ioe);
            }

            session.response(HttpStatus.SC_OK, new StringEntity(
                sbw.toString(),
                StandardCharsets.UTF_8));
        }
    }

    private static final class ActualDropShardCallback
        implements FutureCallback<Object>
    {
        private final DropShardContext contenxt;

        public ActualDropShardCallback(
            final DropShardContext contenxt)
        {
            this.contenxt = contenxt;
        }

        @Override
        public void failed(final Exception ex) {
            contenxt.logger().log(Level.WARNING, "Drop failed", ex);
            completed(new ResetShardCallback(
                contenxt,
                true,
                "Drop failed " + ex.getMessage()));
        }

        @Override
        public void cancelled() {
            completed(new ResetShardCallback(
                contenxt,
                true,
                "Drop cancelled"));
        }

        @Override
        public void completed(final Object result) {
            completed(
                new ResetShardCallback(contenxt, false, "OK"));
        }


        public void completed(final ResetShardCallback callback) {
            contenxt.logger().warning(
                "Resetting shards " + contenxt.shards()
                    + " " + contenxt.queues()
                    + " on " + contenxt.hostname());

            MultiFutureCallback<Object> mfcb =
                new MultiFutureCallback<>(callback);

            // use search client here for retries
            AsyncClient client =
                contenxt.project().resetShardClient().adjust(
                    contenxt.session().context());
            for (ShardDroppableCheckResult check: contenxt.shards()) {
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(
                        "/reset_shard?shard=" + check.shard().id()
                            + "&service=" + check.shard().queueName());

                client.execute(
                    contenxt.consumerHost(),
                    generator,
                    EmptyAsyncConsumerFactory.ANY_GOOD,
                    contenxt.session().listener().createContextGeneratorFor(
                        client),
                    mfcb.newCallback());
            }

            mfcb.done();
        }
    }

    private static class DropShardContext extends DropContext {
        private final int luceneShard;
        private final String dropPolicy;

        public DropShardContext(
            final String hostname,
            final int basePort,
            final List<ShardDroppableCheckResult> shards,
            final int luceneShard,
            final DefaultPsProject project,
            final ProxySession session,
            final Set<String> queues,
            final String dropPolicy)
        {
            super(hostname, basePort, shards, project, session, queues);
            this.luceneShard = luceneShard;
            this.dropPolicy = dropPolicy;
        }

        public int luceneShard() {
            return luceneShard;
        }

        public String dropPolicy() {
            return dropPolicy;
        }
    }
}
