package ru.yandex.mail.search.web.health;

import java.io.IOException;
import java.util.Map;

import org.apache.http.message.BasicHttpRequest;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.config.RetriesConfigBuilder;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.config.ImmutablePsProjectConfig;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.ProjectRoot;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardGroup;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.mail.search.web.health.update.MetricUpdateContext;
import ru.yandex.mail.search.web.health.update.MetricUpdateTaskFactory;
import ru.yandex.mail.search.web.health.update.UpdateTasksManager;
import ru.yandex.mail.search.web.searchmap.SearchmapHelper;
import ru.yandex.parser.config.ConfigException;

public class HealthCheckService implements GenericAutoCloseable<IOException> {
    protected final UpdateTasksManager tasksManager;
    protected final SharedConnectingIOReactor healthReactor;
    protected final AsyncClient healthClient;
    protected final DefaultPsProject project;
    protected final ProjectRoot projectRoot;

    //CSOFF: MagicNumber
    public HealthCheckService(
        final DefaultPsProject project,
        final SearchmapHelper helper,
        final ImmutablePsProjectConfig config)
        throws IOException
    {
        this.project = project;

        healthReactor = new SharedConnectingIOReactor(
            project.webApi().config(),
            project.webApi().config().dnsConfig());

        HttpTargetConfigBuilder builder = new HttpTargetConfigBuilder();
        builder.ioRetries(new RetriesConfigBuilder().count(3).interval(1000));
        builder.timeout(40000);
        builder.poolTimeout(1000);
        builder.connectTimeout(20000);
        builder.connections(10);
        builder.keepAlive(false);

        try {
            ImmutableHttpTargetConfig targetConfig = builder.build();
            this.healthClient =
                project.webApi().registerClient(
                    config.projectId() + "-Health",
                    new AsyncClient(healthReactor, targetConfig),
                    targetConfig);
        } catch (ConfigException ce) {
            throw new IOException(ce);
        }

        projectRoot =
            new ProjectRoot(project, helper);

        tasksManager = new UpdateTasksManager(
            project.webApi(),
            project.logger(),
            helper.dcs(),
            config.projectId(),
            config.metricsUpdateWorkers());

        PrefixedLogger logger =
            project.webApi().config().loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/health/worker")));

        MetricUpdateContext context = new MetricUpdateContext(
            tasksManager,
            healthClient,
            logger);

        for (MetricUpdateTaskFactory factory: config.metrics()) {
            factory.createPerProject(context, projectRoot);
        }

        for (Map.Entry<String, ProjectQueue> queueEntry
            : projectRoot.queues().entrySet())
        {
            ProjectQueue queue = queueEntry.getValue();
            for (MetricUpdateTaskFactory factory: config.metrics()) {
                factory.createPerQueue(context, queue);
            }

            for (ShardGroup shardGroup: queue.groups()) {
                if (shardGroup.node().zk().isEmpty()) {
                    logger.info("Skipping shard group, no zk" + shardGroup.id());
                    continue;
                }

                for (MetricUpdateTaskFactory factory: config.metrics()) {
                    factory.createPerShardGroup(context, shardGroup);
                }

                for (Shard shard: shardGroup.shards()) {
                    for (Map.Entry<String, ShardReplica> replica
                        : shard.replicas().entrySet())
                    {
                        for (MetricUpdateTaskFactory factory
                            : config.metrics())
                        {
                            factory.createPerShardReplica(
                                context,
                                replica.getValue());
                        }
                    }
                }
            }
        }
    }

    public UpdateTasksManager tasksManager() {
        return tasksManager;
    }

    public DefaultPsProject project() {
        return project;
    }

    public ProjectRoot projectRoot() {
        return projectRoot;
    }

    public void start() {
        this.healthReactor.start();
        this.healthClient.start();
        tasksManager.start();
    }

    @Override
    public void close() throws IOException {
        this.tasksManager.close();
        this.healthReactor.close();
        this.healthClient.close();
    }

    //CSON: MagicNumber
}
