package ru.yandex.mail.search.web.health;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardGroup;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class HealthInfoHandler implements ProxyRequestHandler {
    private static final String NONMTN_POSTFIX = "search.yandex.net";
    private static final Set<String> METRICS_NAMES =
        Collections.unmodifiableSet(
            new LinkedHashSet<>(
                Arrays.asList("CopyShardCheck", "Halted")));

    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER =
        new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);

    private final HealthCheckService service;

    public HealthInfoHandler(final HealthCheckService service) {
        this.service = service;
    }

    protected void handleEmpty(
        final ProxySession session,
        final Collection<ProjectQueue> queues)
        throws IOException, BadRequestException
    {
        Set<String> metrics = session.params().get("metrics", METRICS_NAMES, SET_PARSER);
        int limit = session.params().getInt("limit", 100);
        List<ShardWrapper> shards = new ArrayList<>(65534 * queues.size());
        for (ProjectQueue queue: queues) {
            for (ShardGroup group: queue.groups()) {
                for (Shard shard: group.shards()) {
                    shards.add(new ShardWrapper(shard, metrics));
                }
            }
        }

        shards.sort(
            (o1, o2) -> Float.compare(o2.score(), o1.score()));

        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
            writer.startArray();
            for (int i = 0; i < Math.min(limit, shards.size()); i++) {
                ShardWrapper wrapper = shards.get(i);
                Shard shard = wrapper.shard();
                writer.startObject();
                writer.key("shard");
                writer.value(shard.id());
                writer.key("queue");
                writer.value(shard.queueName());
                writer.key("score");
                writer.value(shard.errorScore());
                writer.key("dead-percent");
                writer.value(wrapper.score() * 100.0);
                writer.key("hosts");
                writer.startArray();
                for (ShardReplica replica: shard.replicasCollection()) {
                    writer.value(replica.host().hostname());
                }
                writer.endArray();
                writer.key("metrics");
                writer.startArray();
                for (ShardMetrica metric: shard.metrics().values()) {
                    metric.writeValue(writer);
                }
                writer.endArray();
                writer.endObject();
            }
            writer.endArray();
        }

        session.response(HttpStatus.SC_OK, new StringEntity(
            sbw.toString(),
            StandardCharsets.UTF_8));
    }

    protected void handleByShardId(
        final ProxySession session,
        final Collection<ProjectQueue> queues,
        final int shardId)
        throws HttpException, IOException
    {
        final Map<String, Map<Integer, List<ShardReplica>>> result
            = new LinkedHashMap<>();

        for (ProjectQueue queue: queues) {
            Shard shard = queue.shard(shardId);
            if (shard != null) {
                for (ShardReplica replica: shard.replicas().values()) {
                    result.computeIfAbsent(
                        replica.host().hostname(),
                        (h) -> new LinkedHashMap<>())
                        .computeIfAbsent(shard.id(),
                            (k) -> new ArrayList<>())
                        .add(replica);
                }
            }
        }

        outputOneShard(session, result);
    }

    protected void handleByHost(
        final ProxySession session,
        final Collection<ProjectQueue> queues,
        final String host)
        throws IOException
    {
        String normHost = host;
        int index = host.indexOf(NONMTN_POSTFIX);
        if (index > 1) {
            normHost = host.substring(0, index - 1) + '-';
        }

        session.logger().info("Lookup for " + normHost);

        final Map<String, Map<Integer, List<ShardReplica>>> result
            = new LinkedHashMap<>();

        for (ProjectQueue queue: queues) {
            for (ShardGroup group: queue.groups()) {
                for (Shard shard: group.shards()) {
                    for (Map.Entry<String, ShardReplica> entry
                        : shard.replicas().entrySet())
                    {
                        if (entry.getKey().startsWith(normHost)) {
                            result.computeIfAbsent(
                                entry.getKey(),
                                (h) -> new LinkedHashMap<>())
                                .computeIfAbsent(entry.getValue().shard().id(),
                                    (k) -> new ArrayList<>())
                                .add(entry.getValue());
                        }
                    }
                }
            }
        }

        outputOneShard(session, result);
    }

    protected void outputOneShard(
        final ProxySession session,
        final Map<String, Map<Integer, List<ShardReplica>>> result)
        throws IOException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
            writer.startObject();
            writer.key("hosts");
            writer.startArray();
            for (Map.Entry<String, Map<Integer, List<ShardReplica>>> entry
                : result.entrySet())
            {
                writer.startObject();
                writer.key("hostname");
                writer.value(entry.getKey());
                writer.key("yasm");
                writer.value(service.project.hostYasmPanel(entry.getKey()));
                writer.key("shards");
                writer.startArray();
                for (Map.Entry<Integer, List<ShardReplica>> shardReplicaEntry
                    : entry.getValue().entrySet())
                {
                    writer.startObject();
                    writer.key("shard");
                    writer.value(shardReplicaEntry.getKey());
                    writer.key("queues");
                    writer.startArray();
                    for (ShardReplica replica: shardReplicaEntry.getValue()) {
                        writer.startObject();
                        writer.key("queue");
                        writer.value(
                            replica.shard().shardGroup().queue().service());
                        writer.key("shard");
                        writer.value(replica.shard().id());
                        writer.key("metrics");
                        writer.value(replica.metrics());
                        writer.endObject();
                    }
                    writer.endArray();
                    writer.endObject();

                }
                writer.endArray();
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
        }

        session.response(HttpStatus.SC_OK, new StringEntity(
            sbw.toString(),
            StandardCharsets.UTF_8));
    }
    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        String request = session.params().getString("request");
        if (request.isEmpty()) {
            handleEmpty(session, service.projectRoot().queues().values());
            return;
        }

        Integer shardId;
        try {
            Long longValue = Long.parseLong(request);
            shardId = (int) (longValue % SearchMap.SHARDS_COUNT);
        } catch (NumberFormatException nfe) {
            handleByHost(
                session,
                service.projectRoot().queues().values(),
                request.trim().toLowerCase(Locale.ENGLISH));
            return;
        }

        handleByShardId(
            session,
            service.projectRoot().queues().values(),
            shardId);
    }

    private static final class ShardWrapper {
        private final Shard shard;
        private final float score;

        public ShardWrapper(
            final Shard shard,
            final Set<String> metrics)
        {
            this.shard = shard;

            this.score = shard.errorScore(metrics);
        }

        public float score() {
            return score;
        }

        public Shard shard() {
            return shard;
        }
    }
}
