package ru.yandex.mail.search.web.health.base;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.mail.search.web.WebtoolsProject;
import ru.yandex.mail.search.web.searchmap.MailSearchHost;
import ru.yandex.mail.search.web.searchmap.SearchmapHelper;

public class ProjectRoot {
    private final Map<String, ProjectQueue> queues;
    private final Map<MailSearchHost, List<ShardReplica>> replicasByHost;
    private final WebtoolsProject project;

    public ProjectRoot(
        final WebtoolsProject project,
        final SearchmapHelper searchmapHelper)
    {
        this.project = project;
        Map<MailSearchHost, List<ShardReplica>> replicasByHost
            = new LinkedHashMap<>();

        Map<String, ProjectQueue> queues = new LinkedHashMap<>();
        for (String service: searchmapHelper.services()) {
            ProjectQueue queue =
                new ProjectQueue(
                    searchmapHelper.serviceMap().get(service),
                    service,
                    this);
            //
            int groups = 0;
            for (ShardGroup group: queue.groups()) {
                if (group.node().zk().isEmpty()) {
                    project.logger().info("Skipping shard group, no zk" + group.id());
                    continue;
                }

                groups += 1;
                for (Shard shard: group.shards()) {
                    for (ShardReplica replica: shard.replicasCollection()) {
                        replicasByHost.computeIfAbsent(
                            replica.host(),
                            (h) -> new ArrayList<>()).add(replica);
                    }
                }
            }
            if (groups > 0) {
                queues.put(service, queue);
            } else {
                project.logger().info("Skipping queue, no groups " + queue.service());
            }
        }

        this.replicasByHost = Collections.unmodifiableMap(replicasByHost);
        this.queues = Collections.unmodifiableMap(queues);
    }

    public Map<String, ProjectQueue> queues() {
        return queues;
    }

    public Set<String> queueNames() {
        return queues.keySet();
    }

    public WebtoolsProject project() {
        return project;
    }

    public List<ShardReplica> replicas(final MailSearchHost host) {
        return replicasByHost.get(host);
    }

    public ProjectQueue queue(final String service) {
        return queues.get(service);
    }
    public Map<MailSearchHost, List<ShardReplica>> replicasByHost() {
        return replicasByHost;
    }
}
