package ru.yandex.mail.search.web.health.base;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.mail.search.web.health.DcAwareHostname;
import ru.yandex.mail.search.web.health.Metrica;
import ru.yandex.mail.search.web.health.ShardMetrica;
import ru.yandex.mail.search.web.searchmap.MailSearchHost;
import ru.yandex.mail.search.web.searchmap.SearchmapShardGroup;

public final class Shard {
    private final ShardGroup shardGroup;
    private final int id;
    private final Map<String, ShardReplica> replicas;
    private final Map<String, List<ShardReplica>> replicasByDc;
    private final Map<String, ShardMetrica> metrics;

    public Shard(
        final ShardGroup shardGroup,
        final int id,
        final SearchmapShardGroup node)
    {
        this.id = id;
        this.shardGroup = shardGroup;

        Map<String, List<ShardReplica>> dcGroopedReplicas = new LinkedHashMap<>();
        Map<String, ShardReplica> replicas = new LinkedHashMap<>();
        for (MailSearchHost host: node.hosts()) {
            ShardReplica replica = new ShardReplica(this, host);
            replicas.put(host.hostname(), replica);
            String dc = DcAwareHostname.extractDc(host.hostname());
            dcGroopedReplicas.computeIfAbsent(dc, (k) -> new ArrayList<>(2)).add(replica);
        }

        this.replicas = Collections.unmodifiableMap(replicas);
        this.metrics = new LinkedHashMap<>();
        this.replicasByDc = Collections.unmodifiableMap(dcGroopedReplicas);
    }

    public ShardReplica replica(final String hostname) {
        return this.replicas.get(hostname);
    }

    public int id() {
        return id;
    }

    public ShardGroup shardGroup() {
        return shardGroup;
    }

    public Map<String, ShardReplica> replicas() {
        return replicas;
    }

    public Collection<ShardReplica> replicasCollection() {
        return replicas.values();
    }

    public ProjectQueue queue() {
        return shardGroup.queue();
    }

    public String queueName() {
        return queue().service();
    }

    protected void addReplicaMetric(final Metrica metrica) {
        metrics.computeIfAbsent(
            metrica.type(),
            (k) -> new ShardMetrica(
                metrica.type() + ':' + id(),
                metrica.type())).add(metrica);
    }

    public synchronized float errorScore() {
        float score = 0;
        for (ShardMetrica metric: metrics.values()) {
            score += metric.errorScore();
        }

        return score;
    }

    public synchronized float errorScore(final Set<String> metrics) {
        int dcFailedCount = 0;
        int dcCount = this.replicasByDc.size();
        if (dcCount == 0) {
            return 1;
        }

        for (Map.Entry<String, List<ShardReplica>> entry: this.replicasByDc.entrySet()) {
            boolean dcFailed = true;
            for (ShardReplica replica: entry.getValue()) {
                boolean replicaFailed = false;
                for (Metrica metrica: replica.metrics()) {
                    if (metrics.contains(metrica.type())) {
                        if (metrica.status() != PropertyStatus.OK) {
                            replicaFailed = true;
                            break;
                        }
                    }
                }

                if (!replicaFailed) {
                    dcFailed = false;
                    break;
                }
            }
            if (dcFailed) {
                dcFailedCount += 1;
            }
        }

        return ((float) dcFailedCount) / dcCount;
    }

    public synchronized Map<String, ShardMetrica> metrics() {
        return metrics;
    }
}
