package ru.yandex.mail.search.web.health.metrics;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpHost;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonStreamAsyncConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.config.check.copyshard.ImmutableCopyShardConfig;
import ru.yandex.mail.search.web.health.BasicMetrica;
import ru.yandex.mail.search.web.health.base.ProjectRoot;
import ru.yandex.mail.search.web.health.base.ShardGroup;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.mail.search.web.health.update.MetricUpdateContext;
import ru.yandex.mail.search.web.health.update.MetricUpdateTask;
import ru.yandex.mail.search.web.health.update.MetricUpdateTaskFactory;
import ru.yandex.mail.search.web.searchmap.MailSearchHost;

public class CopyShardsMetricFactory implements MetricUpdateTaskFactory {
    private static final String DESC_PREFIX = "Скопированность - ";

    private final ImmutableCopyShardConfig config;

    public CopyShardsMetricFactory(final ImmutableCopyShardConfig config) {
        this.config = config;
    }


    @Override
    public void createPerProject(
        final MetricUpdateContext context, final ProjectRoot root)
    {
        for (Map.Entry<MailSearchHost, List<ShardReplica>> entry
            : root.replicasByHost().entrySet())
        {
            PrefixedLogger logger = context.logger().addPrefix(
                "CopyShard-" + config.name() +
                    +'-' + entry.getKey().hostname().substring(0, 8)
                    + '-' + entry.getKey().searchPort());
            List<ShardReplica> withZks =
                entry.getValue().stream()
                    .filter((r) -> !r.shard().shardGroup().node().zk().isEmpty())
                    .collect(Collectors.toList());

            context.tasksManager().addTask(
                new CopyShardsMetricUpdateTask(
                    withZks,
                    entry.getKey(),
                    context.client(),
                    logger));
        }
    }

    @Override
    public void createPerShardGroup(
        final MetricUpdateContext context,
        final ShardGroup group)
    {
        if (config.services().isEmpty()) {
            if (!group.queue().service().equalsIgnoreCase(
                group.queue().projectRoot().project().defaultService()))
            {
                return;
            }
        } else if (!config.services().contains(group.queue().service())) {
            return;
        }

        for (MailSearchHost host: group.node().hosts()) {

        }
    }

    private class CopyShardsMetricUpdateTask implements MetricUpdateTask {
        private final MailSearchHost host;
        private final AsyncClient client;
        private final PrefixedLogger logger;
        private final AtomicReference<Future<List<JsonMap>>> futureRef
            = new AtomicReference<>();
        private final BasicAsyncRequestProducerGenerator generator;
        private final HttpHost httpHost;
        private final Map<Integer, List<BasicMetrica>> properties;
        private final BasicMetrica updateMetric;

        public CopyShardsMetricUpdateTask(
            final List<ShardReplica> replicas,
            final MailSearchHost host,
            final AsyncClient client,
            final PrefixedLogger logger)
        {
            this.host = host;
            this.client = client;
            this.logger =
                logger.addPrefix("CopyShardCheck-" + host.hostname());
            int port;
            if (config.dumpPortOffset() >= 0) {
                port = host.basePort() + config.dumpPortOffset();
            } else {
                port = host.dumpPort();
            }
            this.httpHost = new HttpHost(host.hostname(), port);
            this.generator =
                new BasicAsyncRequestProducerGenerator(
                    httpHost.toString()
                        + "?listjobs&json=true");

            Map<Integer, List<BasicMetrica>> properties = new LinkedHashMap<>();
            for (ShardReplica replica: replicas) {
                BasicMetrica metric =
                    new BasicMetrica(
                        "CopyShard status " + name(),
                        "CopyShard",
                        DESC_PREFIX + "No data");
                replica.addMetric(metric);
                properties.computeIfAbsent(
                    replica.shard().id(),
                    (k) -> new ArrayList<>()).add(metric);
            }

            updateMetric =
                new BasicMetrica(
                    "CopyShardCheck",
                    "CopyShardCheck");

            this.properties = Collections.unmodifiableMap(properties);
        }

        @Override
        public PrefixedLogger logger() {
            return logger;
        }

        @Override
        public String name() {
            return host.hostname();
        }

        @Override
        public String hostname() {
            return host.hostname();
        }

        @Override
        public void cancel() {
            Future<List<JsonMap>> future = futureRef.get();
            if (future != null) {
                future.cancel(true);
            }
        }

        protected void fillError(final String error) {
            for (Map.Entry<Integer, List<BasicMetrica>> entry
                : properties.entrySet())
            {
                for (BasicMetrica property: entry.getValue()) {
                    property.error(error);
                }
            }
        }

        protected List<String> processs()
            throws
            JsonException,
            ExecutionException,
            TimeoutException,
            InterruptedException
        {
            Future<List<JsonMap>> future = client.execute(
                httpHost,
                generator,
                new JsonStreamAsyncConsumerFactory<>(JsonObject::asMap),
                EmptyFutureCallback.INSTANCE);
            futureRef.set(future);

            List<String> errors = new ArrayList<>();
            List<JsonMap> result;

            try {
                result = future.get(config.timeout(), TimeUnit.MILLISECONDS);
            } catch (ExecutionException | TimeoutException ee) {
                future.cancel(true);
                logger.warning(
                    "Cannot get from " + httpHost.toString() + " "
                        + generator.toString());
                throw ee;
            }

            if (result.size() == 0) {
                fillError("No lucene shards found");
                return errors;
            }

            JsonMap first = result.get(0);
            int shards = first.getList("shards_status").size();

            boolean[] statuses = new boolean[shards];
            Arrays.fill(statuses, true);

            for (JsonMap copyJob : result) {
                JsonList shardsStatuses =
                    copyJob.getList("shards_status");
                for (int i = 0; i < shardsStatuses.size(); i++) {
                    statuses[i] &= "FINISHED".equalsIgnoreCase(
                        shardsStatuses.get(i).asString());
                }
            }

            for (Map.Entry<Integer, List<BasicMetrica>> entry
                : properties.entrySet())
            {
                int luceneShard = entry.getKey() % shards;

                for (BasicMetrica property: entry.getValue()) {
                    if (statuses[luceneShard]) {
                        property.ok("FINISHED");
                    } else {
                        property.error(
                            "at least one in PENDING state");
                    }
                }
            }

            return errors;
        }

        @Override
        public void run() {
            logger.info("Running check");
            List<String> errors = Collections.emptyList();
            try {
                errors = processs();
                logger.info("Check finished");
            } catch (Exception e) {
                logger.log(Level.WARNING, "Task failed", e);
            }

            if (errors.isEmpty()) {
                updateMetric.ok("OK");
            } else {
                updateMetric.error(String.join(",", errors));
            }
        }
    }
}
