package ru.yandex.mail.search.web.health.metrics;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import org.apache.http.HttpHost;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.config.check.halted.ImmutableHaltedCheckConfig;
import ru.yandex.mail.search.web.health.BasicMetrica;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.ProjectRoot;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardGroup;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.mail.search.web.health.update.MetricUpdateContext;
import ru.yandex.mail.search.web.health.update.MetricUpdateTask;
import ru.yandex.mail.search.web.health.update.MetricUpdateTaskFactory;
import ru.yandex.mail.search.web.searchmap.MailSearchHost;

public class HaltedShardMetricFactory implements MetricUpdateTaskFactory {
    private final ImmutableHaltedCheckConfig config;

    public HaltedShardMetricFactory(
        final ImmutableHaltedCheckConfig config)
    {
        this.config = config;
    }

    @Override
    public void createPerProject(
        final MetricUpdateContext context, final ProjectRoot root)
    {
        Map<MailSearchHost, Map<String, Holder>> map
            = new LinkedHashMap<>();

        for (Map.Entry<String, ProjectQueue> queue: root.queues().entrySet()) {
            for (ShardGroup group: queue.getValue().groups()) {
                if (group.node().zk().isEmpty()) {
                    context.logger().info("Skipping shard group, no zk" + group.id());
                    continue;
                }
                for (Shard shard: group.shards()) {
                    for (ShardReplica replica: shard.replicasCollection()) {
                        Holder holder = map.computeIfAbsent(
                            replica.host(),
                            (h) -> new LinkedHashMap<>())
                            .computeIfAbsent(
                                queue.getKey(),
                                (k) -> new Holder(group));

                        BasicMetrica property =
                            new BasicMetrica(
                                "Halted " + queue.getKey(),
                                "Halted",
                                "Halted check - No data");
                        replica.addMetric(property);
                        holder.add(shard.id(), property);
                    }
                }
            }
        }

        for (Map.Entry<MailSearchHost, Map<String, Holder>> entry
            : map.entrySet())
        {
            context.tasksManager().addTask(
                new HaltedShardMetrictUpdateTask(
                    entry.getValue(),
                    entry.getKey(),
                    context.client(),
                    context.logger()));
        }
    }

    private static final class Holder {
        private final ShardGroup group;
        private final Map<Integer, BasicMetrica> propertyMap;

        public Holder(
            final ShardGroup group)
        {
            this.group = group;
            this.propertyMap = new LinkedHashMap<>();
        }

        public void add(
            final Integer shard,
            final BasicMetrica property)
        {
            this.propertyMap.put(shard, property);
        }

        public Map<Integer, BasicMetrica> props() {
            return propertyMap;
        }
    }

    private class HaltedShardMetrictUpdateTask
        implements MetricUpdateTask
    {
        private final Map<String, Holder> holders;
        private final MailSearchHost host;
        private final AsyncClient client;
        private final PrefixedLogger logger;
        private final AtomicReference<Future<JsonObject>> futureRef
            = new AtomicReference<>();
        private final BasicAsyncRequestProducerGenerator generator;
        private final HttpHost httpHost;

        public HaltedShardMetrictUpdateTask(
            final Map<String, Holder> holderMap,
            final MailSearchHost host,
            final AsyncClient client,
            final PrefixedLogger logger)
        {
            this.holders = Collections.unmodifiableMap(holderMap);
            this.host = host;
            this.client = client;
            this.logger = logger.addPrefix("HaltedCheck-" + host.hostname());
            int port;
            if (config.consumerPortOffset() >= 0) {
                port = host.basePort() + config.consumerPortOffset();
            } else {
                port = host.consumerPort();
            }
            this.httpHost = new HttpHost(host.hostname(), port);

            this.generator =
                new BasicAsyncRequestProducerGenerator(
                    "/status?verbose=false");
        }

        @Override
        public PrefixedLogger logger() {
            return logger;
        }

        @Override
        public String name() {
            return null;
        }

        @Override
        public String hostname() {
            return host.hostname();
        }

        @Override
        public void cancel() {
            Future<JsonObject> future = futureRef.get();
            if (future != null) {
                future.cancel(true);
            }
        }

        @Override
        public void run() {
            logger.info("Running check");
            Future<JsonObject> future = client.execute(
                httpHost,
                generator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                EmptyFutureCallback.INSTANCE);
            futureRef.set(future);

            try {
                JsonObject result =
                    future.get(config.timeout(), TimeUnit.MILLISECONDS);
                logger.info("Data received");
                JsonList haltedList =
                    result.asMap().getListOrNull("halted-shards");
                Map<String, Set<Integer>> haltedMap = new LinkedHashMap<>();

                if (haltedList != null) {
                    for (JsonObject jo: haltedList) {
                        String line = jo.asString();
                        String[] split = line.split("/");
                        String queue = split[0];
                        Integer shard = Integer.parseInt(split[1]);
                        haltedMap.computeIfAbsent(
                            queue,
                            (q) -> new LinkedHashSet<>()).add(shard);
                    }
                }

                for (Map.Entry<String, Holder> entry: holders.entrySet()) {
                    Set<Integer> halted = haltedMap.get(entry.getKey());
                    for (Map.Entry<Integer, BasicMetrica> propEntry
                        : entry.getValue().props().entrySet())
                    {
                        if (halted != null
                            && halted.contains(propEntry.getKey()))
                        {
                            propEntry.getValue().error("Halted");
                        } else {
                            propEntry.getValue().ok("OK");
                        }
                    }
                }
            } catch (Exception e) {
                logger.log(
                    Level.WARNING,
                    "Failed update halted shards " + httpHost,
                    e);
                future.cancel(true);
            }

            logger.info("Check finished");
        }
    }
}
