package ru.yandex.mail.search.web.health.metrics;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import org.apache.http.HttpHost;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.config.check.peach.ImmutablePeachMetricConfig;
import ru.yandex.mail.search.web.config.check.peach.PeachMetricConfigDefaults;
import ru.yandex.mail.search.web.health.BasicMetrica;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardGroup;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.mail.search.web.health.update.MetricUpdateContext;
import ru.yandex.mail.search.web.health.update.MetricUpdateTask;
import ru.yandex.mail.search.web.health.update.MetricUpdateTaskFactory;
import ru.yandex.mail.search.web.searchmap.MailSearchHost;

public class PeachQueueMetricFactory implements MetricUpdateTaskFactory {
    private static final String DESC_PREFIX = "Очередь Пича - ";

    private final ImmutablePeachMetricConfig config;

    public PeachQueueMetricFactory(final ImmutablePeachMetricConfig config) {
        this.config = config;
    }

    public ImmutablePeachMetricConfig config() {
        return config;
    }

    @Override
    public void createPerShardGroup(
        final MetricUpdateContext context,
        final ShardGroup group)
    {
        if (config.services().isEmpty()) {
            if (!group.queue().service().equalsIgnoreCase(
                group.queue().projectRoot().project().defaultService()))
            {
                return;
            }
        } else if (!config.services().contains(group.queue().service())) {
            return;
        }

        for (MailSearchHost host: group.node().hosts()) {
            for (String queue: config.queues()) {
                PrefixedLogger logger = context.logger().addPrefix(
                    "Peach-" + config.name() + '-' + queue + '-' + group.id()
                        + '-' + host.hostname().substring(0, 8));

                context.tasksManager().addTask(
                    new DefaultPeachQueuelenUpdateTask(
                        group,
                        host,
                        context.client(),
                        queue,
                        logger));
            }
        }
    }

    private class DefaultPeachQueuelenUpdateTask implements MetricUpdateTask {
        private final ShardGroup group;
        private final MailSearchHost host;
        private final AsyncClient client;
        private final PrefixedLogger logger;
        private final AtomicReference<Future<JsonObject>> futureRef
            = new AtomicReference<>();
        private final BasicAsyncRequestProducerGenerator generator;
        private final HttpHost httpHost;
        private final Map<Integer, BasicMetrica> properties;
        private final BasicMetrica updateMetric;

        public DefaultPeachQueuelenUpdateTask(
            final ShardGroup group,
            final MailSearchHost host,
            final AsyncClient client,
            final String queue,
            final PrefixedLogger logger)
        {
            this.group = group;
            this.host = host;
            this.client = client;
            this.logger =
                logger.addPrefix("PeachQueueCheck-" + host.hostname());
            this.httpHost = new HttpHost(host.hostname(), host.searchPort());
            String uri = httpHost.toString() + config.queuelenBaseUri();
            if (PeachMetricConfigDefaults.DEFAULT_PEACH_QUEUE.equals(queue)) {
                uri += "peach_url:*+AND+NOT+peach_queue:*";
            } else {
                uri += "peach_queue:*%23" + queue;
            }

            this.generator =
                new BasicAsyncRequestProducerGenerator(uri);

            Map<Integer, BasicMetrica> properties = new LinkedHashMap<>();
            for (Shard shard: group.shards()) {
                ShardReplica replica = shard.replica(host.hostname());
                BasicMetrica metric =
                    new BasicMetrica(
                        "Peach queuelen " + name() + ' ' + queue,
                        "Peach",
                        DESC_PREFIX + queue + " No data");
                properties.put(shard.id(), metric);
                replica.addMetric(metric);
            }

            updateMetric =
                new BasicMetrica(
                    "PeachUpdateStatus",
                    "Peach");

            group.addMetric(updateMetric);
            this.properties = Collections.unmodifiableMap(properties);
        }

        @Override
        public PrefixedLogger logger() {
            return logger;
        }

        @Override
        public String name() {
            return group.queue().service();
        }

        @Override
        public String hostname() {
            return host.hostname();
        }

        @Override
        public void cancel() {
            Future<JsonObject> future = futureRef.get();
            if (future != null) {
                future.cancel(true);
            }
        }

        @Override
        public void run() {
            Future<JsonObject> future = client.execute(
                httpHost,
                generator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                EmptyFutureCallback.INSTANCE);
            futureRef.set(future);

            List<String> errors = new ArrayList<>();
            Set<Integer> updated = new LinkedHashSet<>(properties.size());
            JsonObject result;
            StringBuilder sb = new StringBuilder();
            try {
                result = future.get(config.timeout(), TimeUnit.MILLISECONDS);
                JsonMap map = result.asMap();
                for (Map.Entry<String, JsonObject> entry: map.entrySet()) {
                    int sepIndex = entry.getKey().indexOf('#');
                    long freq = entry.getValue().asMap().getLong("freq");
                    if (sepIndex <= 0) {
                        String error =
                            "Invalid shard number in peach " + entry.getKey();
                        logger.warning(error);
                        errors.add(error);
                        continue;
                    }

                    String shardStr = entry.getKey().substring(0, sepIndex);
                    Integer shard;
                    try {
                        shard = Integer.parseInt(shardStr);
                    } catch (NumberFormatException nfe) {
                        String error =
                            "Peach returned bad shard id " + shardStr;
                        logger.warning(error);
                        errors.add(error);
                        continue;
                    }

                    BasicMetrica property = properties.get(shard);
                    if (property == null) {
                        String error =
                            "Peach shard out of range "+ entry.getKey();
                        logger.warning(error);
                        errors.add(error);
                        continue;
                    }

                    updated.add(shard);
                    if (freq < config.threshold()) {
                        sb.append(" ");
                        sb.append(shard);
                        sb.append(" ok");
                        property.ok(String.valueOf(freq));
                    } else {
                        sb.append(" ");
                        sb.append(shard);
                        sb.append(" error");
                        property.error(DESC_PREFIX +
                            String.valueOf(freq)
                                + " > " + config.threshold());
                    }
                }

                for (Map.Entry<Integer, BasicMetrica> prop
                    : properties.entrySet())
                {
                    if (!updated.contains(prop.getKey())) {
                        prop.getValue().ok("0");
                    }
                }

            } catch (Exception e) {
                logger.log(Level.WARNING, "Task failed", e);
                future.cancel(true);
            }

            if (errors.isEmpty()) {
                updateMetric.ok("OK");
            } else {
                updateMetric.error(String.join(",", errors));
            }
        }
    }
}
