package ru.yandex.mail.search.web.health.update;

import java.io.IOException;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.logging.Level;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.WebApi;
import ru.yandex.mail.search.web.health.DcAwareHostname;
import ru.yandex.mail.search.web.health.Metrica;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class UpdateTasksManager
    implements GenericAutoCloseable<IOException>
{
    private static final String UNKNOWN_DC = "unknown";

    private static final long HOST_REQUEST_INTERVAL = 20000L;
    private static final long TASK_REPEAT_INTERVAL = 60000L;

    private final PrefixedLogger logger;
    private final Map<String, Worker[]> workers;

    public UpdateTasksManager(
        final WebApi webApi,
        final PrefixedLogger logger,
        final Set<String> dcs,
        final String name,
        final int threads)
    {
        this.logger = logger;

        Set<String> allDcs = new LinkedHashSet<>(dcs);
        allDcs.add(UNKNOWN_DC);

        ThreadGroup group = new ThreadGroup("MetrictUpdater");
        workers = new LinkedHashMap<>(dcs.size());
        for (String dc: allDcs) {
            TimeFrameQueue<Long> taskLag = new TimeFrameQueue<>(webApi.config().metricsTimeFrame());
            webApi.registerStater(
                new PassiveStaterAdapter<>(
                    taskLag,
                    new DuplexStaterFactory<>(
                        new NamedStatsAggregatorFactory<>(
                            dc + '-' + name + "-task_lag_axxx",
                            new MaxAggregatorFactory(0L)),
                        new NamedStatsAggregatorFactory<>(
                            dc + '-' + name + "-task_lag_ammm",
                            CountAggregatorFactory.INSTANCE))));
            Worker[] workers = new Worker[threads];
            for (int i = 0; i < threads; i++) {
                workers[i] = new Worker(group, taskLag, name + '-' + dc + "-Updater-" + i);
            }
            this.workers.put(dc, workers);
        }
    }

    public void start() {
        for (Map.Entry<String, Worker[]> entry: workers.entrySet()) {
            logger.info(
                "Starting health task manager, workers "
                    + entry.getValue().length + " in " + entry.getKey());
            for (Worker worker: entry.getValue()) {
                worker.start();
            }
        }
    }

    public void addTask(final MetricUpdateTask task) {
        DcAwareHostname hostname = new DcAwareHostname(task.hostname());
        UpdateTaskWrapper wrapper = new UpdateTaskWrapper(task);
        Worker[] workers = this.workers.get(hostname.dc());
        if (workers == null) {
            workers = this.workers.get(UNKNOWN_DC);
        }

        workers[Math.abs(task.hostname().hashCode()) % workers.length].add(wrapper);
    }

    private static class UpdateTaskWrapper implements Runnable {
        private final MetricUpdateTask task;
        private long lastStartTs = 0L;
        private long nextStartTs = 0L;

        public UpdateTaskWrapper(final MetricUpdateTask task) {
            this.task = task;
        }

        @Override
        public void run() {
            lastStartTs = System.currentTimeMillis();
            try {
                task.run();
            } finally {
                nextStartTs =
                    System.currentTimeMillis()
                        + Math.max(
                        TASK_REPEAT_INTERVAL,
                        task.updateInterval());
            }
        }

        public void delay(final long ts) {
            nextStartTs = ts;
        }

        public String hostname() {
            return task.hostname();
        }

        @Override
        public String toString() {
            return task.name() + "@" + task.hostname();
        }

        public PrefixedLogger logger() {
            return task.logger();
        }
    }

    private static class TaskStartTsComparator
        implements Comparator<UpdateTaskWrapper>
    {
        @Override
        public int compare(
            final UpdateTaskWrapper o1,
            final UpdateTaskWrapper o2)
        {
            return Long.compare(o1.nextStartTs, o2.nextStartTs);
        }
    }

    private class Worker extends Thread implements Runnable {
        private volatile boolean stopped = false;
        private final PriorityBlockingQueue<UpdateTaskWrapper> workerQueue
            = new PriorityBlockingQueue<>(
                1000,
            new TaskStartTsComparator());
        private final Map<String, Long> hostsLastRequest
            = new LinkedHashMap<>();
        private final TimeFrameQueue<Long> taskLag;

        public Worker(final ThreadGroup group, final TimeFrameQueue<Long> taskLag, final String name) {
            super(group, name);
            this.setDaemon(true);
            this.taskLag = taskLag;
        }

        public void add(final UpdateTaskWrapper task) {
            this.workerQueue.add(task);
        }

        @Override
        public void run() {
            logger.info("Starting " + workerQueue.toString());
            try {
                while (!stopped) {
                    UpdateTaskWrapper task = workerQueue.poll();
                    if (task == null) {
                        Thread.sleep(1000);
                        continue;
                    }

                    long ts = System.currentTimeMillis();

                    long diff = task.nextStartTs - ts;
                    if (diff > 0) {
                        Thread.sleep(diff);
                        ts = System.currentTimeMillis();
                    } else {
                        taskLag.accept(-diff);
                        task.logger().warning(
                            "Task exec lag is "
                                + Metrica.timeToHuman(-diff));
                    }

                    try {
                        long hostLastRequest =
                            hostsLastRequest.getOrDefault(task.hostname(), 0L);
                        if (hostLastRequest + HOST_REQUEST_INTERVAL <= ts) {
                            hostsLastRequest.put(task.hostname(), ts);
                            task.logger().fine("Starting task");
                            task.run();
                            task.logger().fine("Task finished");
                        } else {
                            long delay =
                                hostLastRequest + HOST_REQUEST_INTERVAL - ts;
                            task.logger().warning(
                                "Delaying task, due to host limiter for "
                                    + Metrica.timeToHuman(delay));

                            task.delay(
                                hostLastRequest + HOST_REQUEST_INTERVAL);
                        }
                    } catch (Exception e) {
                        task.logger().log(
                            Level.WARNING,
                            "Task finished with error " + task,
                            e);
                    } finally {
                        workerQueue.add(task);
                    }
                }
            } catch (InterruptedException ie) {
                ie.printStackTrace();
            }
        }
    }

    @Override
    public void close() throws IOException {
        for (Worker[] workers: this.workers.values()) {
            for (Worker worker: workers) {
                worker.interrupt();
            }
        }
    }
}
