package ru.yandex.mail.search.web.iex;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumer;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.mail.search.web.WebApi;

public class RemorphHandler implements HttpAsyncRequestHandler<Object> {
    private final WebApi proxy;

    public RemorphHandler(final WebApi proxy) {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<Object> processRequest(
        final HttpRequest httpRequest,
        final HttpContext httpContext)
        throws HttpException
    {
        if (!(httpRequest instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncDomConsumer(
            ((HttpEntityEnclosingRequest) httpRequest).getEntity());
    }

    @Override
    public void handle(
        final Object payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        BasicProxySession session =
            new BasicProxySession(proxy, exchange, context);
        try {
            if (payload instanceof Map) {
                Map<?, ?> json = ValueUtils.asMap(payload);
                Object rules = json.get("rules");
                Object text = json.get("text");
                if (!(rules instanceof String) || !(text instanceof String)) {
                    session.response(
                        HttpStatus.SC_BAD_REQUEST,
                        "rules and text must be a string");
                    return;
                }
                String rulesStr = (String) rules;
                String textStr = (String) text;
                if (rulesStr.isEmpty() || textStr.isEmpty()) {
                    session.response(
                        HttpStatus.SC_BAD_REQUEST,
                        "rules and text are required");
                    return;
                }
                String remorphResult = execRemorph(session, rulesStr, textStr);
                session.response(HttpStatus.SC_OK, remorphResult);
            }
        } catch (JsonUnexpectedTokenException e) {
            String msg = "Failed to handle request, error: ";
            session.logger().log(Level.SEVERE, msg, e);
            session.response(HttpStatus.SC_BAD_REQUEST, msg + e.getMessage());
        }
    }

    private String execRemorph(
        final ProxySession session,
        final String rules,
        final String text)
        throws IOException
    {
        File rulesFile = null;
        File textFile = null;
        try {
            StringBuilder builder = new StringBuilder();
            String dir = "remorph/tmp/";
            File dirFile = new File(dir);
            rulesFile = File.createTempFile("rules-", ".remorph", dirFile);
            session.logger().info("Rules file created: " + rulesFile.getName());
            try (FileOutputStream os = new FileOutputStream(rulesFile)) {
                os.write(rules.getBytes(StandardCharsets.UTF_8));
                os.flush();
            }
            textFile = File.createTempFile("text-", ".html", dirFile);
            session.logger().info("Text file created: " + textFile.getName());
            try (FileOutputStream os = new FileOutputStream(textFile)) {
                os.write(text.getBytes(StandardCharsets.UTF_8));
                os.flush();
            }
            String command = "remorph/bin/remorphparser -h full -r "
                + dir + rulesFile.getName() + ' ' + dir + textFile.getName();
            session.logger().info("Start remorph with command: " + command);
            Process process = Runtime.getRuntime().exec(command);
            try (BufferedReader in = new BufferedReader(
                new InputStreamReader(
                    process.getInputStream(),
                    StandardCharsets.UTF_8)))
            {
                String line;
                while ((line = in.readLine()) != null) {
                    builder.append(line).append('\n');
                }
            }
            try (BufferedReader in = new BufferedReader(
                new InputStreamReader(
                    process.getErrorStream(),
                    StandardCharsets.UTF_8)))
            {
                String line;
                while ((line = in.readLine()) != null) {
                    builder.append(line).append('\n');
                }
            }
            return builder.toString();
        } finally {
            String errorMsg = "Unable to delete tmp file ";
            if (rulesFile != null) {
                if (rulesFile.delete()) {
                    session.logger().info("Rules file deleted");
                } else {
                    session.logger().warning(errorMsg + rulesFile.getName());
                }
            }
            if (textFile != null) {
                if (textFile.delete()) {
                    session.logger().info("Text file deleted");
                } else {
                    session.logger().warning(errorMsg + textFile.getName());
                }
            }
        }
    }
}
