package ru.yandex.mail.search.web.images;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.webtools.mail.MailSearchProject;
import ru.yandex.search.document.mail.MailMetaInfo;

public class ImagesHandler implements ProxyRequestHandler {
    private static final String UID = "uid";
    private static final String MDB = "mdb";
    private static final String EXTRACT = "extract";
    private static final String PG = "pg";
    private static final String FIRST = "first";
    private static final String COUNT = "count";
    private static final String REQUEST = "request";
    private static final String SEARCH_FILTER = "search-filter";

    private final MailSearchProject webApi;

    public ImagesHandler(final MailSearchProject webApi) {
        this.webApi = webApi;
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        Long uid = session.params().getLong(UID);
        AsyncClient client =
            webApi.msearchProxyClient().adjust(session.context());
        QueryConstructor qc =
            new QueryConstructor("/api/async/mail/search?");
        qc.append(UID, uid);
        qc.append(MDB, PG);
        qc.append(FIRST, session.params().getLong(FIRST));
        qc.append(COUNT, session.params().getLong(COUNT));
        qc.append(REQUEST, session.params().getString(REQUEST));

        for (String filter: session.params().getAll(SEARCH_FILTER)) {
            qc.append(SEARCH_FILTER, filter);
        }

        client.execute(
            webApi.config().mailSearch().host(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().adjustContextGenerator(
                client.httpClientContextGenerator()),
            new MailSearchCallback(session, uid));
    }

    private final class MailSearchCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Long uid;

        private MailSearchCallback(final ProxySession session, final Long uid) {
            super(session);
            this.uid = uid;
        }

        @Override
        public void completed(final JsonObject serp) {
            try {
                StringBuilder uri =
                    new StringBuilder("/facts?&mdb=pg&uid=" + uid);
                Map<String, JsonMap> mids = new LinkedHashMap<>();

                JsonList envelopes = serp.asMap().getList("envelopes");
                for (JsonObject envelopeObj : envelopes) {
                    JsonMap envelope = envelopeObj.asMap();
                    String mid =
                        envelope.getString(MailMetaInfo.MID, null);
                    if (mid != null) {
                        mids.put(mid, envelope);
                    }

                    uri.append("&mid=");
                    uri.append(mid);
                }

                if (this.session.params().getBoolean(
                    EXTRACT,
                    false))
                {
                    uri.append("&extract");
                }

                IexCallback callback = new IexCallback(session, mids);
                if (mids.size() <= 0) {
                    callback.completed(JsonMap.EMPTY);
                } else {
                    AsyncClient client =
                        webApi.iexClient().adjust(session.context());
                    client.execute(
                        webApi.config().iex().host(),
                        new BasicAsyncRequestProducerGenerator(uri.toString()),
                        JsonAsyncTypesafeDomConsumerFactory.OK,
                        session.listener().adjustContextGenerator(
                            client.httpClientContextGenerator()),
                        callback);
                }
            } catch (JsonException | BadRequestException e) {
                failed(e);
            }
        }
    }

    private static final class IexCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Map<String, JsonMap> mids;

        private IexCallback(
            final ProxySession session,
            final Map<String, JsonMap> mids)
            throws BadRequestException
        {
            super(session);

            this.mids = mids;
        }

        private boolean isNews(final JsonMap fact) throws JsonException {
            for (String key: fact.keySet()) {
                if (key.startsWith("taksa_widget_type_")) {
                    if ("news".equalsIgnoreCase(
                        fact.getString(key, null)))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        @Override
        public void completed(final JsonObject resp) {
            StringBuilderWriter sbw = new StringBuilderWriter();

            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startArray();
                JsonMap root = resp.asMap();
                for (Map.Entry<String, JsonMap> midEntry: mids.entrySet()) {
                    String mid = midEntry.getKey();
                    List<JsonObject> facts = root.getListOrNull(mid);
                    if (facts == null) {
                        session.logger().info(
                            "Mid " + mid + " has not facts");
                        continue;
                    }

                    for (JsonObject factObj: facts) {
                        JsonMap fact = factObj.asMap();
                        if (!isNews(fact)) {
                            continue;
                        }

                        JsonMap meta = midEntry.getValue();
                        JsonMap from =
                            meta.getList(MailMetaInfo.FROM).get(0).asMap();
                        String fromStr =
                            from.getString("local")
                                + '@' + from.getString("domain");

                        writer.startObject();
                        writer.key(MailMetaInfo.MID);
                        writer.value(mid);
                        writer.key(MailMetaInfo.FROM);
                        writer.value(fromStr);
                        writer.key(MailMetaInfo.SUBJECT);
                        writer.value(meta.getString(MailMetaInfo.SUBJECT));

                        JsonMap img = fact.getMapOrNull("selected_img");
                        if (img != null) {
                            writer.key("img");
                            writer.value(img.getString("src"));
                        }

                        writer.endObject();
                    }
                }
                writer.endArray();
            } catch (JsonException | IOException e) {
                failed(e);
            }

            session.response(HttpStatus.SC_OK, sbw.toString());
        }
    }
}
