package ru.yandex.mail.search.web.info;

import java.util.AbstractMap;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.LongPair;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.config.ImmutableFilterSearchConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.webtools.mail.MailSearchExtractSession;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.SearchProxyParams;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class CheckIndexExtractor implements InfoExtractor {
    private static final int MAX_MISSING_LENGTH = 10;

    private final User user;
    private final AsyncClient fsClient;
    private final ImmutableFilterSearchConfig fsConfig;
    private final FutureCallback<Map.Entry<String, JsonObject>> callback;
    private final MailSearchExtractSession session;
    private final AsyncClient searchClient;
    private final Supplier<? extends HttpClientContext> contextSupplier;
    private final boolean fetchMissing;
    private final String luceneBase;

    public CheckIndexExtractor(
        final MailSearchExtractSession session,
        final FutureCallback<Map.Entry<String, JsonObject>> callback)
        throws BadRequestException
    {
        this.callback = callback;
        this.session = session;

        Long uid = session.params().getLong(InfoHandler.UID);
        String service = session.params().getString(SearchProxyParams.SERVICE);
        this.user = new User(service, new LongPrefix(uid));

        fsClient = session.project().filterSearchClient();
        fsConfig = session.project().config().filterSearch();

        this.searchClient =
            session.project().searchClient().adjust(
                session.session().context());
        this.contextSupplier
            = session.session().listener().adjustContextGenerator(
            searchClient.httpClientContextGenerator());

        this.fetchMissing =
            session.params().getBoolean(
                "fetch-missing-mids",
                false);

        StringBuilder luceneBase =
            new StringBuilder("/search?&get=mid&prefix=");
        luceneBase.append(user.prefix());
        luceneBase.append("&service=");
        luceneBase.append(user.service());
        this.luceneBase = luceneBase.toString();
    }

    @Override
    public void execute() {
        StringBuilder sb = new StringBuilder(fsConfig.uri().getScheme());
        sb.append("://");
        sb.append(fsConfig.uri().getHost());
        sb.append(':');
        sb.append(fsConfig.uri().getPort());

        StringBuilder fcUri = new StringBuilder(sb);
        fcUri.append("/folders_counters?&caller=msearch&uid=");
        fcUri.append(user.prefix().toStringFast());

        StringBuilder revisionUri = new StringBuilder(sb);
        revisionUri.append("/mailbox_revision?caller=msearch&uid=");
        revisionUri.append(user.prefix().toStringFast());

        MultiFutureCallback<JsonObject> mfcb =
            new MultiFutureCallback<>(
                new FilterSearchCallback(callback));
        fsClient.execute(
            fsConfig.host(),
            new BasicAsyncRequestProducerGenerator(fcUri.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.session().listener().adjustContextGenerator(
                fsClient.httpClientContextGenerator()),
            mfcb.newCallback());

        fsClient.execute(
            fsConfig.host(),
            new BasicAsyncRequestProducerGenerator(revisionUri.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.session().listener().adjustContextGenerator(
                fsClient.httpClientContextGenerator()),
            mfcb.newCallback());

        mfcb.done();
    }

    private final class FilterSearchCallback
        extends AbstractFilterFutureCallback
        <List<JsonObject>, Map.Entry<String, JsonObject>>
    {
        private FilterSearchCallback(
            final FutureCallback<? super Map.Entry<String, JsonObject>> cb)
        {
            super(cb);
        }

        @Override
        public void completed(final List<JsonObject> objects) {
            try {
                LinkedHashMap<String, Long> folders = new LinkedHashMap<>();
                JsonObject revisionObj = objects.get(1);
                long revision =
                    revisionObj.asMap().getLong("mailbox_revision");
                session.session().logger().info("Revision " + revision);

                JsonMap foldersStat = objects.get(0).asMap().getMap("folders");

                if (foldersStat.size() == 0) {
                    callback.completed(null);
                    return;
                }

                String luceneRequest =
                    luceneBase + "&length=1&text=hid:0+AND+fid:";

                MultiFutureCallback<List<SearchResult>> mfcb =
                    new MultiFutureCallback<>(
                        new AggregateCallback(callback, folders));

                for (Map.Entry<String, JsonObject> folderEntry
                    : foldersStat.entrySet())
                {
                    folders.put(
                        folderEntry.getKey(),
                        folderEntry.getValue().asMap().getLong("cnt"));
                    List<HttpHost> hosts =
                        session.project().searchmap().searchHosts(user);
                    MultiFutureCallback<SearchResult> bcb =
                        new MultiFutureCallback<>(mfcb.newCallback());

                    for (HttpHost host: hosts) {
                        searchClient.execute(
                            host,
                            new BasicAsyncRequestProducerGenerator(
                                luceneRequest + folderEntry.getKey()),
                            SearchResultConsumerFactory.INSTANCE,
                            contextSupplier,
                            new LoggingLuceneCallback(
                                host,
                                session,
                                bcb.newCallback()));
                    }
                    bcb.done();
                }

                mfcb.done();
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    protected JsonMap buildResult(
        final Map<HttpHost, HostStatus> hostCache,
        final long fsTotal)
    {
        JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);
        result.put("FilterSearch Count", new JsonLong(fsTotal));

        for (Map.Entry<HttpHost, HostStatus> entry: hostCache.entrySet()) {
            result.put(
                entry.getKey().toString(),
                entry.getValue().toJsonMap());
        }

        return result;
    }

    private final class AggregateCallback
        extends AbstractFilterFutureCallback<
        List<List<SearchResult>>, Map.Entry<String, JsonObject>>
    {
        private final LinkedHashMap<String, Long> folders;

        private AggregateCallback(
            final FutureCallback<
                ? super Map.Entry<String, JsonObject>> callback,
            final LinkedHashMap<String, Long> folders)
        {
            super(callback);

            this.folders = folders;
        }

        @Override
        public void completed(final List<List<SearchResult>> lists) {
            Map<HttpHost, HostStatus> hostCache = new LinkedHashMap<>();

            int index = 0;
            long fsTotal = 0L;
            for (Map.Entry<String, Long> folder: folders.entrySet()) {
                fsTotal += folder.getValue();

                for (SearchResult result: lists.get(index)) {
                    if (result.hitsCount() < 0) {
                        continue;
                    }

                    hostCache.computeIfAbsent(
                        result.host(),
                        HostStatus::new).addFolderStat(
                        folder.getKey(),
                        folder.getValue(),
                        result.hitsCount());
                }

                index += 1;
            }

            boolean missing = false;

            if (fetchMissing) {
                MultiFutureCallback<MidsResponse> mfcb =
                    new MultiFutureCallback<>(
                        new AggregateMissingCallback(
                            callback,
                            hostCache,
                            fsTotal));

                String missingRequest =
                    luceneBase + "&text=hid:0&length=10000000";

                for (Map.Entry<HttpHost, HostStatus> entry
                    : hostCache.entrySet())
                {
                    if (entry.getValue().needFetchMissing()) {
                        missing = true;
                        searchClient.execute(
                            entry.getKey(),
                            new BasicAsyncRequestProducerGenerator(
                                missingRequest),
                            SearchResultConsumerFactory.INSTANCE,
                            contextSupplier,
                            new LuceneMidsCallback(
                                mfcb.newCallback(),
                                entry.getKey()));
                    }
                }

                if (missing) {
                    AsyncClient msalClient =
                        session.project().msalClient().adjust(
                            session.session().context());
                    msalClient.execute(
                        session.project().config().msal().host(),
                        new BasicAsyncRequestProducerGenerator(
                            "/user-mids?json-type=dollar&uid=" + user.prefix()),
                        JsonAsyncTypesafeDomConsumerFactory.OK,
                        contextSupplier,
                        new MsalMidsCallback(mfcb.newCallback()));
                    mfcb.done();
                }
            }

            if (!missing) {
                callback.completed(
                    new AbstractMap.SimpleEntry<>(
                        InfoHandler.CHECKINDEX,
                        buildResult(hostCache, fsTotal)));
            }
        }
    }

    private final class AggregateMissingCallback
        extends AbstractFilterFutureCallback<List<MidsResponse>, Map
        .Entry<String, JsonObject>>
    {
        private final Map<HttpHost, HostStatus> hostMap;
        private final long fsTotal;

        private AggregateMissingCallback(
            final FutureCallback<? super Map.Entry<String, JsonObject>>
                callback,
            final Map<HttpHost, HostStatus> hostMap, final long fsTotal)
        {
            super(callback);

            this.hostMap = hostMap;
            this.fsTotal = fsTotal;
        }

        @Override
        public void completed(final List<MidsResponse> sets) {
            MidsResponse msalSet = sets.get(sets.size() - 1);
            for (int i = 0; i < sets.size() - 1; i++) {
                MidsResponse lucene = sets.get(i);
                HostStatus status = hostMap.get(lucene.host);

                for (String item: msalSet.mids) {
                    if (!lucene.mids.contains(item)) {
                        status.missingMid(item);
                    }

                    if (status.missingMids.size() > MAX_MISSING_LENGTH) {
                        break;
                    }
                }
            }

            callback.completed(
                new AbstractMap.SimpleEntry<>(
                    InfoHandler.CHECKINDEX,
                    buildResult(hostMap, fsTotal)));
        }
    }

    private final class MsalMidsCallback
        extends AbstractFilterFutureCallback<JsonObject, MidsResponse>
    {
        private MsalMidsCallback(
            final FutureCallback<? super MidsResponse> callback)
        {
            super(callback);
        }

        @Override
        public void completed(final JsonObject map) {
            Set<String> mids;
            try {
                JsonList rows = map.asMap().getList("rows");
                mids = new LinkedHashSet<>(rows.size());
                for (JsonObject jo: rows) {
                    mids.add(jo.asMap().getString("mid"));
                }

                session.session().logger().info(
                    "Msal returned " + mids.size());
                callback.completed(new MidsResponse(null, mids));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private final class LuceneMidsCallback
        extends AbstractFilterFutureCallback<SearchResult, MidsResponse>
    {
        private final HttpHost host;

        private LuceneMidsCallback(
            final FutureCallback<? super MidsResponse> callback,
            final HttpHost host)
        {
            super(callback);
            this.host = host;
        }

        @Override
        public void completed(final SearchResult result) {
            Set<String> mids = new LinkedHashSet<>();
            for (SearchDocument doc: result.hitsArray()) {
                mids.add(doc.attrs().get(MailIndexFields.MID));
            }

            session.session().logger().info(
                "Lucene returned " + host.toString() + ' ' + mids.size());

            callback.completed(new MidsResponse(host, mids));
        }

        @Override
        public void cancelled() {
            super.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            callback.completed(new MidsResponse(host, Collections.emptySet()));
        }
    }

    private final class HostStatus {
        private final HttpHost host;
        private final Map<String, LongPair<Long>> foldersMessageCount;
        private final Set<String> missingMids;

        private HostStatus(final HttpHost host) {
            this.host = host;
            this.foldersMessageCount = new LinkedHashMap<>();
            this.missingMids = new LinkedHashSet<>();
        }

        private void addFolderStat(
            final String folder,
            final long fsStat,
            final long luceneStat)
        {
            this.foldersMessageCount.put(
                folder,
                new LongPair<>(luceneStat, fsStat));
        }

        public void missingMid(final String mid) {
            QueryConstructor qc = new QueryConstructor(BASE_URI);
            try {
                qc.append(
                    "request_text",
                    "&mid=" + mid + "&uid=" + user.prefix().toString());
            } catch (BadRequestException bre) {
                bre.printStackTrace();
            }

            this.missingMids.add(link(qc.toString(), mid));
        }

        public HttpHost host() {
            return host;
        }

        public Map<String, LongPair<Long>> foldersMessageCount() {
            return foldersMessageCount;
        }

        private LongPair<Long> total() {
            long totalFs = 0;
            long totalLucene = 0;
            for (LongPair<Long> value: foldersMessageCount.values()) {
                totalLucene += value.first();
                totalFs += value.second();
            }

            return new LongPair<>(totalLucene, totalFs);
        }

        private JsonMap toJsonMap() {
            JsonMap root = new JsonMap(BasicContainerFactory.INSTANCE);
            for (Map.Entry<String, LongPair<Long>> entry
                : foldersMessageCount.entrySet())
            {
                LongPair<Long> pair = entry.getValue();
                String value =
                    "" + pair.first() + '(' + pair.second() + ')';

                root.put(entry.getKey(), new JsonString(value));
            }

            if (missingMids.size() > 0) {
                JsonList list = new JsonList(BasicContainerFactory.INSTANCE);
                for (String mid: missingMids) {
                    list.add(new JsonString(mid));
                }

                root.put("Missing", list);
            }

            LongPair<Long> pair = total();
            String value =
                "" + pair.first() + '(' + pair.second() + ')';
            root.put("Total", new JsonString(value));

            return root;
        }

        private boolean needFetchMissing() {
            LongPair<Long> pair = total();
            return pair.first() != pair.second();
        }
    }

    private static final class MidsResponse {
        private final HttpHost host;
        private final Set<String> mids;

        private MidsResponse(final HttpHost host, final Set<String> mids) {
            this.host = host;
            this.mids = mids;
        }
    }
}
