package ru.yandex.mail.search.web.info;

import java.net.URISyntaxException;
import java.util.AbstractMap;
import java.util.Map;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.config.FilterSearchConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.ps.webtools.mail.MailSearchExtractSession;
import ru.yandex.search.document.mail.MailMetaInfo;

public class FilterSearchExtractor implements InfoExtractor {
    private static final String RECEIVE_DATE = "receiveDate";
    private final Long uid;
    private final String mid;
    private final MailSearchExtractSession session;
    private final AsyncClient client;
    private final FilterSearchConfig config;
    private final String uri;
    private final FutureCallback<Map.Entry<String, JsonObject>> callback;

    public FilterSearchExtractor(
        final MailSearchExtractSession session,
        final FutureCallback<Map.Entry<String, JsonObject>> callback)
        throws BadRequestException
    {
        this.callback = callback;
        this.session = session;
        this.uid = session.params().getLong(InfoHandler.UID);
        this.mid = session.params().getString(InfoHandler.MID);
        StringBuilder sb;

        client = session.project().filterSearchClient();
        config = session.project().config().filterSearch();
        sb = new StringBuilder(config.uri().toASCIIString());

        sb.append("&order=default&full_folders_and_labels=1&mdb=pg&uid=");
        sb.append(uid);
        sb.append("&caller=msearch&mids=");
        sb.append(mid);
        this.uri = sb.toString();
        session.session().logger().info("FilterSearch " + uri);
    }

    @Override
    public void execute() {
        FilterSearchCallback callback =
            new FilterSearchCallback(this.callback);
        try {
            client.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(uri),
                    new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                        session.project().filterSearchTicket())),
                JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
                session.session().listener()
                    .createContextGeneratorFor(client),
                callback);
        } catch (URISyntaxException e) {
            callback.failed(e);
        }
    }

    private final class FilterSearchCallback
        implements FutureCallback<JsonObject>
    {
        private final FutureCallback<Map.Entry<String, JsonObject>> callback;

        private FilterSearchCallback(
            final FutureCallback<Map.Entry<String, JsonObject>> callback)
        {
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject root) {
            try {
                JsonList envelopes = root.asMap().getList("envelopes");
                JsonMap result = JsonMap.EMPTY;
                if (envelopes.size() > 0) {
                    result = envelopes.get(0).asMap();
                    double ts =
                        result.getDouble(
                            RECEIVE_DATE,
                            -1.0);
                    String stId = result.getString(MailMetaInfo.STID, null);
                    if (stId != null) {
                        result.replace(
                            MailMetaInfo.STID,
                            new JsonString(stidLink(stId)));
                    }

                    if (ts > 0) {
                        result.put(
                            RECEIVE_DATE,
                            new JsonString(unixtimestamp(ts)));
                    }
                }
                callback.completed(
                    new AbstractMap.SimpleEntry<>(
                        InfoHandler.FILTER_SEARCH,
                        result));
            } catch (JsonException je) {
                failed(je);
            }
        }

        protected JsonMap error(final String error) {
            JsonMap host = new JsonMap(BasicContainerFactory.INSTANCE);
            host.put("FilterSearchError", new JsonString(error));
            return host;
        }

        @Override
        public void failed(final Exception e) {
            callback.completed(
                new AbstractMap.SimpleEntry<>(
                    InfoHandler.FILTER_SEARCH,
                    error(e.toString())));
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }
}
