package ru.yandex.mail.search.web.info;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.parser.searchmap.User;
import ru.yandex.ps.webtools.mail.MailSearchExtractSession;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class MailLuceneExtractor
    implements InfoExtractor, FutureCallback<List<SearchResult>>
{
    protected final MailSearchExtractSession session;
    protected final boolean allbackends;
    protected final Set<HttpHost> userHosts;
    protected final User user;
    protected final Long uid;
    protected final String mid;
    protected final String service;
    protected final FutureCallback<Map.Entry<String, JsonObject>> callback;

    public MailLuceneExtractor(
        final MailSearchExtractSession session,
        final FutureCallback<Map.Entry<String, JsonObject>> callback)
        throws BadRequestException
    {
        this.session = session;
        allbackends =
            session.params().getBoolean(
                "lucene-allbackends",
                false);
        this.uid = session.params().getLong("uid");
        this.mid = session.params().getString("mid");
        this.service =
            session.params().getString(
                "service",
                service(uid, session.project().config()));

        user = new User(service, new LongPrefix(uid));
        session.session().logger().info("User " + user.toString());
        userHosts =
            new LinkedHashSet<>(session.searchMap().searchHosts(user));
        session.session().logger().info(
            "User hosts " + userHosts.toString());
        this.callback = callback;
    }

    protected String uri() {
        return "/search?prefix=" + user.prefix()
            + "&service=" + user.service()
            + "&get=*&text=mid_p:" + mid;
    }

    protected String name() {
        return InfoHandler.LUCENE;
    }

    @Override
    public void execute() {
        Supplier<? extends HttpClientContext> contextSupplier
            = session.session().listener().adjustContextGenerator(
                session.project().searchClient().httpClientContextGenerator());

        if (userHosts.isEmpty()) {
            failed(new Exception("No backends found for " + user.toString()));
            return;
        }

        session.session().logger().info("Lucene request " + uri());
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(uri());
        if (!allbackends) {
            PlainUniversalSearchProxyRequestContext context =
                new PlainUniversalSearchProxyRequestContext(
                    user,
                    null,
                    true,
                    session.project().searchClient(),
                    session.session().logger());

            session.project().webApi().parallelRequest(
                session.session(),
                context,
                generator,
                SearchResultConsumerFactory.OK,
                contextSupplier,
                new LoggingLuceneCallback(
                    null,
                    session,
                    new MailLuceneSingleCallback(this)));
        } else {
            AsyncClient searchClient =
                session.project().searchClient().adjust(
                    session.session().context());
            MultiFutureCallback<SearchResult> mfcb =
                new MultiFutureCallback<>(this);
            for (HttpHost host: userHosts) {
                searchClient.execute(
                    host,
                    generator,
                    SearchResultConsumerFactory.INSTANCE,
                    contextSupplier,
                    new LoggingLuceneCallback(
                        host,
                        session,
                        mfcb.newCallback()));
            }
            mfcb.done();
        }
    }

    @Override
    public void completed(final List<SearchResult> immutableResults) {
        JsonMap root = new JsonMap(BasicContainerFactory.INSTANCE);
        List<SearchResult> results = new ArrayList<>(immutableResults);

        results.sort(
            Comparator.comparingLong(SearchResult::zooQueueId).reversed());

        Set<HttpHost> leftHosts = new LinkedHashSet<>(userHosts);
        for (SearchResult result: results) {
            if (result == null) {
                continue;
            }
            JsonMap host = new JsonMap(BasicContainerFactory.INSTANCE);
            host.put(InfoHandler.HITS_COUNT, new JsonLong(result.hitsCount()));
            JsonList list = new JsonList(BasicContainerFactory.INSTANCE);

            for (SearchDocument doc: result.hitsArray()) {
                JsonMap attrs = new JsonMap(BasicContainerFactory.INSTANCE);
                for (Map.Entry<String, String> entry
                    : doc.attrs().entrySet())
                {
                    String value = entry.getValue();
                    if (MailIndexFields.QUEUE_ID.equalsIgnoreCase(
                        entry.getKey()))
                    {
                        value = queueIdLink(user, value);
                    } else if (MailIndexFields.RECEIVED_DATE.equalsIgnoreCase(
                        entry.getKey()))
                    {
                        value = unixtimestamp(value);
                    } else if (MailIndexFields.STID.equalsIgnoreCase(
                        entry.getKey()))
                    {
                        value = stidLink(value);
                    }

                    attrs.put(
                        entry.getKey(),
                        new JsonString(value));
                }
                list.add(attrs);
            }

            host.put(InfoHandler.HITS_ARRAY, list);
            host.put(
                "QueueId",
                new JsonString(
                    queueIdLink(user, result.zooQueueId())));
            root.put(result.host().toString(), host);
            leftHosts.remove(result.host());
        }

        if (allbackends) {
            for (HttpHost luceneHost: leftHosts) {
                root.put(luceneHost.toString(), errorHostInfo());
            }
        }

        callback.completed(
            new AbstractMap.SimpleEntry<>(name(), root));
    }

    protected static JsonMap errorHostInfo() {
        return errorHostInfo("Unknown error");
    }

    protected static JsonMap errorHostInfo(final String error) {
        JsonMap host = new JsonMap(BasicContainerFactory.INSTANCE);
        host.put(InfoHandler.HITS_COUNT, new JsonLong(-1L));
        host.put(
            InfoHandler.HITS_ARRAY,
            new JsonList(BasicContainerFactory.INSTANCE));
        host.put("Error", new JsonString(error));
        return host;
    }

    @Override
    public void failed(final Exception e) {
        JsonMap errorMap = new JsonMap(BasicContainerFactory.INSTANCE);
        for (HttpHost host: userHosts) {
            errorMap.put(host.toString(), errorHostInfo());
        }
        callback.completed(
            new AbstractMap.SimpleEntry<>(name(), errorMap));
    }

    @Override
    public void cancelled() {
        callback.cancelled();
    }

    private static final class MailLuceneSingleCallback
        extends AbstractFilterFutureCallback<SearchResult, List<SearchResult>>
    {
        private MailLuceneSingleCallback(
            final FutureCallback<? super List<SearchResult>> callback)
        {
            super(callback);
        }

        @Override
        public void completed(final SearchResult result) {
            callback.completed(Collections.singletonList(result));
        }
    }
}
