package ru.yandex.mail.search.web.info;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.jute.BinaryInputArchive;
import org.apache.zookeeper.MultiResponse;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.ZooDefs;

import ru.yandex.dispatcher.common.HttpMessage;
import ru.yandex.dispatcher.common.SerializeUtils;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.searchmap.ZooKeeperAddress;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.SearchProxyParams;

public class QueueExtractor
    implements InfoExtractor, FutureCallback<HttpResponse>
{
    protected static final String QUEUEID = "queueId";

    private static final int SKIP_BYTES = 13;

    private final User user;
    private final ExtractSession session;
    private final Long queueIdStart;
    private final Long shard;
    private final String service;
    private final QueryConstructor uri;
    private final AsyncClient queueClient;
    private final FutureCallback<Map.Entry<String, JsonObject>> callback;

    public QueueExtractor(
        final ExtractSession session,
        final AsyncClient queueClient,
        final FutureCallback<Map.Entry<String, JsonObject>> callback)
        throws BadRequestException
    {
        this.session = session;
        this.callback = callback;
        this.queueClient = queueClient;

        this.shard = session.params().getLong(SearchProxyParams.SHARD);
        this.queueIdStart =
            session.params().getLong(QUEUEID, null);

        int count = session.params().getInt("queue-count", 1);
        this.service =
            session.params().getString(SearchProxyParams.SERVICE);
        PrefixType prefixType = session.searchMap().prefixType(service);
        if (prefixType == PrefixType.STRING) {
            user = new User(service, new FixedHasStringPrefix("", shard.intValue()));
        } else {
            user = new User(service, new LongPrefix(shard));
        }

        uri = new QueryConstructor("/getData3?");
        uri.append(SearchProxyParams.SHARD, shard);
        uri.append(SearchProxyParams.SERVICE, service);
        uri.append("count", count);
        uri.append("id", queueIdStart);
    }

    @Override
    public void execute() {
        List<HttpHost> zks =
            session.searchMap().hosts(user).zk().stream()
                .map(ZooKeeperAddress::host).collect(Collectors.toList());

        if (zks.size() <= 0) {
            failed(new Exception("No zookeepers for " + user.toString()));
            return;
        }

        queueClient.adjust(
            session.session().context()).execute(
                zks,
            new BasicAsyncRequestProducerGenerator(uri.toString()),
            BasicAsyncResponseConsumerFactory.OK,
            session.session().listener().adjustContextGenerator(
                queueClient.httpClientContextGenerator()),
            this);

        session.session().logger().info(
            uri.toString() + ' ' + zks.toString());
    }

    @Override
    public void completed(final HttpResponse response) {
        JsonMap root = new JsonMap(BasicContainerFactory.INSTANCE);

        try {
            BinaryInputArchive bia =
                BinaryInputArchive.getArchive(
                    response.getEntity().getContent());
            MultiResponse mrsp = new MultiResponse();
            mrsp.deserialize(bia, "mrsp");

            Iterator<OpResult> iter = mrsp.iterator();
            long queueid = this.queueIdStart;
            while (iter.hasNext()) {
                OpResult op = iter.next();
                if (op.getType() == ZooDefs.OpCode.getData) {
                    OpResult.GetDataResult gdr = (OpResult.GetDataResult) op;
                    HttpMessage message =
                        SerializeUtils.deserializeHttpMessage(gdr.getData());
                    root.put(
                        String.valueOf(queueid),
                        new JsonString(
                            new String(
                                message.toByteArray(
                                    queueid,
                                    shard.intValue(),
                                    service),
                                StandardCharsets.UTF_8)));
                } else if (op.getType() == ZooDefs.OpCode.error) {
                    OpResult.ErrorResult er = (OpResult.ErrorResult) op;
                    root.put(
                        String.valueOf(queueid),
                        new JsonLong(er.getErr()));
                }
                queueid += 1;
            }
        } catch (IOException ioe) {
            failed(ioe);
            return;
        }

        callback.completed(
            new AbstractMap.SimpleEntry<>(InfoHandler.QUEUE, root));
    }

    @Override
    public void failed(final Exception e) {
        session.session().logger().log(
            Level.WARNING,
            "Failed to parse queue messsage",
            e);
        JsonMap root = new JsonMap(BasicContainerFactory.INSTANCE);
        root.put("error", new JsonString(e.toString()));
        callback.completed(
            new AbstractMap.SimpleEntry<>(InfoHandler.QUEUE, root));
    }

    @Override
    public void cancelled() {
        callback.cancelled();
    }

    private static class FixedHasStringPrefix extends StringPrefix {
        private final int shard;

        public FixedHasStringPrefix(final String prefix, final int shard) {
            super(prefix);

            this.shard = shard;
        }

        @Override
        public long hash() {
            return shard;
        }
    }
}
