package ru.yandex.mail.search.web.info;

import java.util.AbstractMap;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.WebtoolsProject;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.proxy.SearchProxyParams;

public class QueueExtractorFactory implements InfoExtractorFactory {
    public static final int MAX_SHARDS = 65534;

    private static final Pattern ID_P =
        Pattern.compile("[^\\w]id=(\\d+)");
    private static final Pattern ZOO_QUEUE_ID_P =
        Pattern.compile("zoo-queue-id=(\\d+)");
    private static final Pattern QUEUE_ID_P =
        Pattern.compile("queue[\\-]?id=(\\d+)");
    private static final Pattern SHARD_P =
        Pattern.compile("shard=(\\d+)");
    private static final Pattern SERVICE_P =
        Pattern.compile("service=([a-zA-Z\\-_]+)");

    private static final ExtractChain SHARD_EXTRACT_CHAIN =
        new PatternExtractChain(SHARD_P, null);
    private static final ExtractChain SERVICE_EXTRACT_CHAIN =
        new PatternExtractChain(SERVICE_P, null);
    private static final ExtractChain QUEUE_ID_CHAIN =
        new PatternExtractChain(
            ID_P,
            new PatternExtractChain(
                ZOO_QUEUE_ID_P,
                new PatternExtractChain(
                    QUEUE_ID_P,
                    null)));

    private final WebtoolsProject webApi;
    private final AsyncClient queueClient;

    public QueueExtractorFactory(
        final DefaultPsProject webApi)
    {
        this.webApi = webApi;
        this.queueClient = webApi.queueClient();
    }

    @Override
    public void create(
        final ProxySession session,
        final FutureCallback<Map.Entry<String, JsonObject>> callback)
    {
        try {
            String text =
                session.params().getString("request", "")
                    .toLowerCase(Locale.ROOT);
            if (!session.params().getBoolean(InfoHandler.QUEUE, false)) {
                callback.completed(null);
                return;
            }

            Long queueId = session.params().getLong(
                QueueExtractor.QUEUEID,
                null);
            Long shard =
                session.params().getLong(
                    SearchProxyParams.SHARD,
                    null);
            Long uid = session.params().getLong(
                InfoHandler.UID,
                null);
            if (uid == null) {
                String uidStr =
                    InfoExtractorFactory.UID_EXTRACT_CHAIN.extract(text);

                if (uidStr != null) {
                    uid = Long.parseLong(uidStr);
                }
            }

            String defaultService = webApi.defaultService(String.valueOf(uid));

            if (shard == null && uid == null) {
                String shardStr = SHARD_EXTRACT_CHAIN.extract(text);
                if (shardStr != null) {
                    shard = Long.parseLong(shardStr);
                }

                if (shard == null) {
                    throw new BadRequestException(
                        "Prefix, uid or shard required");
                }
            }

            if (shard == null) {
                shard = uid % MAX_SHARDS;
            }

            String service =
                session.params().getString(
                    SearchProxyParams.SERVICE,
                    null);
            session.logger().info("Service from param " + service);
            if (service == null) {
                service = SERVICE_EXTRACT_CHAIN.extract(text);
                session.logger().info("Service from text " + service);
                if (service == null) {
                    service = defaultService;
                    session.logger().info(
                        "SetDEfault service " + defaultService);
                }
            }

            if (queueId == null) {
                String queueIdStr = QUEUE_ID_CHAIN.extract(text);

                if (queueIdStr != null) {
                    queueId = Long.parseLong(queueIdStr);
                }
            }

            if (queueId == null) {
                throw new BadRequestException("QueueId required");
            }

            CgiParams params = new CgiParams(session.params());
            params.replace(SearchProxyParams.SERVICE, service);
            params.replace(SearchProxyParams.SHARD, shard.toString());
            params.replace(QueueExtractor.QUEUEID, queueId.toString());
            ExtractSession extractSession =
                new ExtractSession(webApi, session, params);

            new QueueExtractor(extractSession, queueClient, callback).execute();
        } catch (BadRequestException bre) {
            JsonMap root = new JsonMap(BasicContainerFactory.INSTANCE);
            root.put("error", new JsonString(bre.toString()));
            callback.completed(
                new AbstractMap.SimpleEntry<>(InfoHandler.QUEUE, root));
        }
    }
}
