package ru.yandex.mail.search.web.mail.checkindex;

import java.io.IOException;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.config.ImmutableFilterSearchConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.search.web.mail.checkindex.lucene.LuceneFastDocsPerFolderChecker;
import ru.yandex.mail.search.web.mail.checkindex.lucene.LuceneFastDocsTotalCheck;
import ru.yandex.mail.search.web.mail.checkindex.lucene.LuceneIndexCheck;
import ru.yandex.mail.search.web.mail.checkindex.lucene.LucenePeachDocsCheck;
import ru.yandex.mail.search.web.mail.checkindex.lucene.LuceneSlowDocsTotalCheck;
import ru.yandex.ps.webtools.mail.MailSearchProject;

public class CheckIndexHandler implements ProxyRequestHandler {
    private final MailSearchProject webApi;

    private final List<LuceneIndexCheck> checkers;

    public CheckIndexHandler(final MailSearchProject webApi) {
        this.webApi = webApi;
        checkers = Arrays.asList(
            new LuceneFastDocsTotalCheck(),
            new LuceneFastDocsPerFolderChecker(),
            new LuceneSlowDocsTotalCheck(),
            new LucenePeachDocsCheck());
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        CheckIndexSession checkIndexSession =
            new CheckIndexSession(session, webApi);

        gatherFromDatabase(checkIndexSession);
    }

    private void gatherFromDatabase(final CheckIndexSession session) {
        AsyncClient fsClient;
        ImmutableFilterSearchConfig fsConfig;

        fsClient = session.webApi().filterSearchClient();
        fsConfig = session.webApi().config().filterSearch();

        StringBuilder sb = new StringBuilder(fsConfig.uri().getScheme());
        sb.append("://");
        sb.append(fsConfig.uri().getHost());
        sb.append(':');
        sb.append(fsConfig.uri().getPort());

        StringBuilder fcUri = new StringBuilder(sb);
        fcUri.append("/folders_counters?&caller=msearch&uid=");
        fcUri.append(session.user().prefix().toStringFast());

        StringBuilder revisionUri = new StringBuilder(sb);
        revisionUri.append("/mailbox_revision?caller=msearch&uid=");
        revisionUri.append(session.user().prefix().toStringFast());

        MultiFutureCallback<JsonObject> mfcb =
            new MultiFutureCallback<>(
                new FilterSearchCallback(session));
        BasicAsyncRequestProducerGenerator generator
            = new BasicAsyncRequestProducerGenerator(fcUri.toString());
        generator.addHeader(
            new BasicHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                webApi.filterSearchTicket()));
        fsClient.execute(
            fsConfig.host(),
            generator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.session().listener().adjustContextGenerator(
                fsClient.httpClientContextGenerator()),
            mfcb.newCallback());

        generator = new BasicAsyncRequestProducerGenerator(revisionUri.toString());
        generator.addHeader(
            new BasicHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                webApi.filterSearchTicket()));

        fsClient.execute(
            fsConfig.host(),
            generator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.session().listener().adjustContextGenerator(
                fsClient.httpClientContextGenerator()),
            mfcb.newCallback());

        mfcb.done();
    }

    private final class FilterSearchCallback
        extends AbstractProxySessionCallback<List<JsonObject>>
    {
        private final CheckIndexSession session;

        private FilterSearchCallback(
            final CheckIndexSession session)
        {
            super(session.session());

            this.session = session;
        }

        @Override
        public void completed(final List<JsonObject> objects) {
            try {
                JsonObject revisionObj = objects.get(1);
                long revision =
                    revisionObj.asMap().getLong("mailbox_revision");
                session.logger().info("Revision " + revision);

                JsonMap foldersStat = objects.get(0).asMap().getMap("folders");

                if (foldersStat.size() == 0) {
                    session.callback().failed(
                        new Exception("No folders found for users"));
                    return;
                }

                Map<String, Long> dbPerFolder = new LinkedHashMap<>();
                long dbTotal = 0;
                for (Map.Entry<String, JsonObject> folderEntry
                    : foldersStat.entrySet())
                {
                    String id = folderEntry.getKey();
                    long docs = folderEntry.getValue().asMap().getLong("cnt");
                    dbPerFolder.put(id, docs);
                    dbTotal += docs;
                }

                MultiFutureCallback<Object> mfcb =
                    new MultiFutureCallback<>(session.callback());

                MdbFoldersDocs docs = new MdbFoldersDocs(dbTotal, dbPerFolder);
                for (LuceneIndexCheck checker: checkers) {
                    checker.check(session, docs, mfcb);
                }

                mfcb.done();

                mfcb.done();
            } catch (JsonException | BadRequestException e) {
                failed(e);
            }
        }
    }

    public static String printWithDiff(final long db, final long lucene) {
        String value = Long.toString(lucene);
        long diff = lucene - db;
        if (diff != 0) {
            value += " (<font color=\"red\">";
            if (diff > 0) {
                value += '+';
            }

            value += diff;
            value += "</font>)";
        }

        return value;
    }
}
