package ru.yandex.mail.search.web.mail.checkindex;

import java.util.function.Supplier;

import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.info.InfoExtractorFactory;
import ru.yandex.mail.search.web.info.InfoHandler;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.ps.webtools.mail.MailSearchProject;
import ru.yandex.search.prefix.LongPrefix;

public class CheckIndexSession {
    private final ProxySession session;
    private final MailSearchProject webApi;
    private final User user;
    private final Long uid;
    private final PrefixedLogger logger;
    private final CheckIndexCallback callback;
    private final AsyncClient searchClient;
    private final Supplier<? extends HttpClientContext> contextSupplier;
    private final StringBuilder luceneBase;

    private final boolean corp;

    public CheckIndexSession(
        final ProxySession session,
        final MailSearchProject webapi)
        throws BadRequestException
    {
        this.session = session;
        this.webApi = webapi;

        String text =
            session.params().getString("request", "");

        Long uid = session.params().getLong(
            InfoHandler.UID,
            null);

        session.logger().info("Request " + text);
        if (uid == null) {
            String uidStr =
                InfoExtractorFactory.UID_EXTRACT_CHAIN.extract(text);

            if (uidStr != null) {
                uid = Long.parseLong(uidStr);
            }
        }

        if (uid == null) {
            throw new BadRequestException("Failed to extract uid");
        }

        this.uid = uid;
        this.corp = BlackboxUserinfo.corp(uid);

        String defaultService = webApi.config().mainQueue();

        String service =
            InfoExtractorFactory.SERVICE_EXTRACT_CHAIN.extract(text);

        if (service == null) {
            service = defaultService;
        }

        this.user = new User(service, new LongPrefix(uid));

        this.logger = session.logger();

        this.callback = new CheckIndexCallback(session);

        searchClient = webapi.searchClient().adjust(session.context());
        contextSupplier = session.listener().adjustContextGenerator(
            searchClient.httpClientContextGenerator());

        luceneBase =
            new StringBuilder("/search?&get=mid&prefix=");
        luceneBase.append(user.prefix());
        luceneBase.append("&service=");
        luceneBase.append(user.service());
    }

    public ProxySession session() {
        return session;
    }

    public MailSearchProject webApi() {
        return webApi;
    }

    public User user() {
        return user;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public boolean corp() {
        return corp;
    }

    public CheckIndexCallback callback() {
        return callback;
    }

    public CgiParams params() {
        return session.params();
    }

    public AsyncClient searchClient() {
        return searchClient;
    }

    public Supplier<? extends HttpClientContext> contextSupplier() {
        return contextSupplier;
    }

    public String luceneBase() {
        return luceneBase.toString();
    }

    public Long uid() {
        return uid;
    }
}
