package ru.yandex.mail.search.web.mail.checkindex.lucene;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.mail.search.web.mail.checkindex.CheckIndexHandler;
import ru.yandex.mail.search.web.mail.checkindex.CheckIndexSession;
import ru.yandex.mail.search.web.mail.checkindex.MdbFoldersDocs;
import ru.yandex.mail.search.web.mail.checkindex.RowType;
import ru.yandex.search.proxy.SearchResultConsumerFactory;

public class LuceneFastDocsPerFolderChecker implements LuceneIndexCheck {
    @Override
    public void check(
        final CheckIndexSession session,
        final MdbFoldersDocs docs,
        final MultiFutureCallback<Object> mfcb)
        throws BadRequestException
    {
        if (!session.params().getBoolean("per-folder", false)) {
            return;
        }

        session.logger().info("Per Folder docs stat");

        List<HttpHost> hosts =
            session.webApi().searchmap().searchHosts(session.user());

        AsyncClient searchClient =
            session.webApi().searchClient().adjust(
                session.session().context());
        Supplier<? extends HttpClientContext> contextSupplier
            = session.session().listener().adjustContextGenerator(
            searchClient.httpClientContextGenerator());

        StringBuilder luceneBase =
            new StringBuilder("/search?&get=mid&prefix=");
        luceneBase.append(session.user().prefix());
        luceneBase.append("&service=");
        luceneBase.append(session.user().service());

        String luceneRequest =
            luceneBase.toString() + "&length=1&text=hid:0+AND+fid:";
        for (Map.Entry<String, Long> entry: docs.folderStat().entrySet()) {
            MultiFutureCallback<LuceneDocsCountHostResult> bcb =
                new MultiFutureCallback<>(
                    new FolderCallback(entry, mfcb.newCallback(), session));

            for (HttpHost host: hosts) {
                searchClient.execute(
                    host,
                    new BasicAsyncRequestProducerGenerator(
                        luceneRequest + entry.getKey()),
                    SearchResultConsumerFactory.INSTANCE,
                    contextSupplier,
                    new LuceneCallback(host, bcb.newCallback()));
            }

            bcb.done();
        }
    }

    private static final class FolderCallback
        implements FutureCallback<List<LuceneDocsCountHostResult>>
    {
        private final Map.Entry<String, Long> fidDocs;
        private final FutureCallback<Object> callback;
        private final CheckIndexSession session;

        private FolderCallback(
            final Map.Entry<String, Long> fidDocs,
            final FutureCallback<Object> callback,
            final CheckIndexSession session)
        {
            this.fidDocs = fidDocs;
            this.callback = callback;
            this.session = session;
        }

        @Override
        public void completed(final List<LuceneDocsCountHostResult> entries) {
            Map<String, Object> row = new LinkedHashMap<>();
            row.put("#", "FID " + fidDocs.getKey());
            row.put("DB", fidDocs.getValue());

            for (LuceneDocsCountHostResult entry: entries) {
                if (entry.ok()) {
                    row.put(entry.shortHostname(),
                        CheckIndexHandler.printWithDiff(
                            fidDocs.getValue(),
                            entry.docsCount()));
                } else {
                    row.put(
                        entry.shortHostname(),
                        entry.exception().getMessage());
                }
            }

            session.callback().addRow(RowType.FID, row);
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            callback.completed(null);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }
}
