package ru.yandex.mail.search.web.searchmap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.parser.searchmap.SearchMap;

public class ListShardsWithFilterHandler implements ProxyRequestHandler {
    private final SearchmapHelper helper;
    private final DefaultPsProject webApi;

    public ListShardsWithFilterHandler(
        final DefaultPsProject webApi,
        final SearchmapHelper helper)
    {
        this.helper = helper;
        this.webApi = webApi;
    }

    protected void write(
        final ProxySession session,
        final Stream<SearchmapShardGroup> groupStream)
        throws HttpException, IOException
    {
        int length = session.params().getInt("length", Integer.MAX_VALUE);
        int offset = session.params().getInt("offset", 0);
        JsonType jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter jw = jsonType.create(sbw)) {
            jw.startArray();

            List<SearchmapShardGroup> group =
                groupStream.limit(offset + length).collect(Collectors.toList());
            for (int i = offset; i < Math.min(group.size(), length); i++) {
                jw.startObject();
                jw.value(group.get(i));
                jw.endObject();
            }
            jw.endArray();
        }

        session.response(
            HttpStatus.SC_OK,
            new StringEntity(sbw.toString(), ContentType.APPLICATION_JSON));
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        String service = session.params().getString("service");
        if (!helper.serviceMap().containsKey(service)) {
            session.response(HttpStatus.SC_NOT_FOUND);
            return;
        }

        String shards = session.params().getString("shards", null);
        if (shards != null) {
            int dash = shards.indexOf('-');
            int minShard;
            int maxShard;
            try {
                if (dash < 0) {
                    int shard = Integer.parseInt(shards);
                    maxShard = shard;
                    minShard = shard;
                } else {
                    minShard =
                        Math.max(
                            Integer.parseInt(shards.substring(0, dash)),
                            0);
                    maxShard =
                        Math.min(
                            Integer.parseInt(shards.substring(dash + 1)),
                            (int) (SearchMap.SHARDS_COUNT - 1));
                }

                write(
                    session,
                    helper.serviceMap().get(service).stream().filter(
                        new ShardFilter(minShard, maxShard)));
            } catch (NumberFormatException nfe) {
                throw new BadRequestException(
                    "Bad shards param " + shards,
                    nfe);
            }
        }

        String host = session.params().getString("host", null);
        if (host != null) {
            Map<String, List<SearchmapShardGroup>> hostMap =
                helper.hostnmMap().get(service);

            List<SearchmapShardGroup> result = new ArrayList<>();
            for (Map.Entry<String, List<SearchmapShardGroup>> entry
                : hostMap.entrySet())
            {
                String hostName = entry.getKey();
                if (hostName.contains(host)) {
                    result.addAll(hostMap.get(hostName));
                }
            }

            write(session, result.stream());
        }

        if (host == null && shards == null) {
            throw new BadRequestException("No host or shards specified");
        }
    }

    private final class ShardFilter
        implements Predicate<SearchmapShardGroup>
    {
        private final int start;
        private final int end;

        private ShardFilter(final int shard) {
            this(shard, shard);
        }

        private ShardFilter(final int start, final int end) {
            this.start = start;
            this.end = end;
        }

        @Override
        public boolean test(final SearchmapShardGroup shardGroup) {
            for (int i = start; i <= end; i++) {
                if (shardGroup.shards().contains(i)) {
                    webApi.logger().info(shardGroup.toString());
                    return true;
                }
            }

            return false;
        }
    }
}
