package ru.yandex.mail.search.web.searchmap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.web.health.DcAwareHostname;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapRow;
import ru.yandex.parser.searchmap.SearchMapShard;

public class SearchmapHelper extends TimerTask implements GenericAutoCloseable<IOException> {
    private final int shardsCount;
    private final Supplier<SearchMap> supplier;
    private final Timer timer;
    private final PrefixedLogger logger;

    private Map<String, Map<String, List<SearchmapShardGroup>>> hostnmMap;
    private Map<String, List<SearchmapShardGroup>> serviceMap;
    private Set<String> dcs;
    private int currentVersion = -1;

    public SearchmapHelper(
        final PrefixedLogger logger,
        final Supplier<SearchMap> supplier,
        final int shardsCount)
    {
        this.shardsCount = shardsCount;
        this.supplier = supplier;

        this.logger = logger;
        rebuild(supplier.get());

        timer = new Timer(true);
        timer.schedule(
            this,
            TimeUnit.MINUTES.toMillis(2),
            TimeUnit.MINUTES.toMillis(2));
    }

    @Override
    public void close() throws IOException {
        timer.cancel();
    }

    @Override
    public void run() {
        rebuild(supplier.get());
    }

    private synchronized void rebuild(final SearchMap searchMap) {
        if (searchMap.version() <= currentVersion) {
            return;
        }

        logger.info("Rebuilding searchmap from version " + currentVersion + " to " + searchMap.version());

        Map<String, Map<String, List<SearchmapShardGroup>>> hostnmMap
            = new LinkedHashMap<>();

        Map<String, List<SearchmapShardGroup>> serviceMap
            = new LinkedHashMap<>();

        Set<String> dcs = new LinkedHashSet<>();
        for (String service: searchMap.names()) {
            SearchMapRow row = searchMap.row(service);
            List<SearchmapShardGroup> groups = buildGroups(row, shardsCount);
            serviceMap.putIfAbsent(service, groups);

            Map<String, List<SearchmapShardGroup>> hostMap
                = new LinkedHashMap<>();
            for (SearchmapShardGroup group: groups) {
                for (MailSearchHost host: group.hosts()) {
                    dcs.add(DcAwareHostname.extractDc(host.hostname()));

                    hostMap.computeIfAbsent(
                        host.hostname(),
                        (k) -> new ArrayList<>()).add(group);
                }
            }

            hostnmMap.put(service, hostMap);
        }

        this.dcs = Collections.unmodifiableSet(dcs);
        this.hostnmMap = Collections.unmodifiableMap(hostnmMap);
        this.serviceMap = Collections.unmodifiableMap(serviceMap);
        this.currentVersion = searchMap.version();
    }

    private static List<SearchmapShardGroup> buildGroups(
        final SearchMapRow row,
        final int shardsCount)
    {
        Map<ShardGroupKey, SearchmapShardGroup> groups = new LinkedHashMap<>();
        for (int i = 0; i < shardsCount; i++) {
            SearchMapShard shard = row.get(i);
            final int num = i;
            final ShardGroupKey key = new ShardGroupKey(shard);
            groups.computeIfAbsent(
                key,
                (k) -> new SearchmapShardGroup(
                    shard,
                    num)).updateShardRange(num);
        }

        return new ArrayList<>(groups.values());
    }

    public synchronized Map<String, Map<String, List<SearchmapShardGroup>>> hostnmMap() {
        return hostnmMap;
    }

    public synchronized Map<String, List<SearchmapShardGroup>> serviceMap() {
        return serviceMap;
    }

    public synchronized Set<String> services() {
        return serviceMap.keySet();
    }

    public synchronized Set<String> dcs() {
        return dcs;
    }
}
