package ru.yandex.ps.webtools.aceventura;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;

import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaPrefixParser;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;

public class FastCheckIndexHandler implements ProxyRequestHandler {
    private final AceventuraProject project;

    public FastCheckIndexHandler(
        final AceventuraProject project)
    {
        this.project = project;
    }

    protected void dbEmails(
        final AceVenturaPrefix user,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/get-user-emails?&scope=contacts");
        qc.append("uid", user.uid());
        qc.append("userType", user.userType().lowName());

        AsyncClient msalClient = project.msalClient();
        msalClient.execute(
            project.msalHost(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(msalClient),
            callback);
    }

    protected void dbShared(
        final AceVenturaPrefix user,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/get-shared-owners-lists?&scope=contacts");
        qc.append("uid", user.uid());
        qc.append("userType", user.userType().lowName());
        AsyncClient msalClient = project.msalClient();
        msalClient.execute(
            project.msalHost(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(msalClient),
            callback);
    }

    protected void indexCounters(
        final AceVenturaPrefix user,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor(
            "/printkeys?&service=aceventura_change_log_prod" +
                "&field=av_record_type_p&print-freqs&max-freq=100000&skip-deleted&hr");
        qc.append("prefix", user.toString());

        AsyncClient client = project.searchClient().adjust(session.context());

        client.execute(
            project.searchmap().searchHosts(
                new User("aceventura_change_log_prod", user)),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        AceVenturaPrefix user =
            session.params().get("user", new AceVenturaPrefixParser());

        MultiFutureCallback<JsonObject> mfcb = new MultiFutureCallback<>(new CheckCallback(session, user));
//        DoubleFutureCallback<JsonObject, JsonObject> dfcb =
//            new DoubleFutureCallback<>(new CheckCallback(session, user));

        dbEmails(user, session, mfcb.newCallback());
        dbShared(user, session, mfcb.newCallback());
        indexCounters(user, session, mfcb.newCallback());
        mfcb.done();
    }

    private static final class CheckCallback
        extends AbstractProxySessionCallback<List<JsonObject>>
    {
        private final String contactsPrefix;
        private final String emailsPrefix;
        private final String tagsPrefix;
        private final String sharedPrefix;

        public CheckCallback(final ProxySession session, final AceVenturaPrefix user) {
            super(session);

            this.contactsPrefix = user.prefix() + "#contact";
            this.emailsPrefix = user.prefix() + "#email";
            this.tagsPrefix = user.prefix() + "#tag";
            this.sharedPrefix = user.prefix() + "#shared";
        }

        @Override
        public void completed(final List<JsonObject> list) {
            int indexContacts = 0;
            int indexEmails = 0;
            int indexTags = 0;
            int indexShared = 0;
            int dbEmailsCount = 0;
            int dbSharedCount = 0;

            try {
                JsonList dbEmails = list.get(0).asList();
                for (JsonObject jo: dbEmails) {
                    int atIndex = MailAliases.emailSeparatorPos(jo.asMap().getString("email", ""));
                    if (atIndex <= 0) {
                        continue;
                    }

                    dbEmailsCount += 1;
                }

                JsonList dbShared = list.get(1).asList();
                int dbSharedCnt = dbShared.size();

                JsonMap indexCounters = list.get(2).asMap();
                JsonMap map = indexCounters.getMapOrNull(contactsPrefix);
                if (map != null) {
                    indexContacts = map.getInt("freq", 0);
                }

                map = indexCounters.getMapOrNull(tagsPrefix);
                if (map != null) {
                    indexTags = map.getInt("freq", 0);
                }

                map = indexCounters.getMapOrNull(emailsPrefix);
                if (map != null) {
                    indexEmails = map.getInt("freq", 0);
                }

                map = indexCounters.getMapOrNull(sharedPrefix);
                if (map != null) {
                    indexShared = map.getInt("freq", 0);
                }
            } catch (JsonException | NumberFormatException je) {
                failed(je);
                return;
            }

            session.logger().info("Emails " + indexEmails + " Contacts: " + indexContacts + " Tags " + indexTags  + " shared " + indexShared);
            AceVenturaCheckIndexReport report =
                new AceVenturaCheckIndexReport(
                    Math.abs(dbEmailsCount - indexEmails),
                    0,
                    0,
                    Math.abs(dbSharedCount - indexShared));

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
                writer.value(report);
            } catch (IOException ioe) {
                failed(ioe);
            }

            session.response(
                HttpStatus.SC_OK,
                new StringEntity(sbw.toString(), StandardCharsets.UTF_8));
        }
    }

    private static class AceVenturaCheckIndexReport implements JsonValue {
        private final int score;

        private final int emailsDiff;
        private final int contactsDiff;
        private final int tagsDiff;
        private final int sharedDiff;

        public AceVenturaCheckIndexReport(
            final int emailsDiff,
            final int contactsDiff,
            final int tagsDiff,
            final int sharedDiff)
        {
            this.emailsDiff = emailsDiff;
            this.contactsDiff = contactsDiff;
            this.tagsDiff = tagsDiff;
            this.sharedDiff = sharedDiff;
            this.score = this.emailsDiff + this.contactsDiff + this.tagsDiff;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("score");
            writer.value(score);
            writer.key("emailsDiff");
            writer.value(emailsDiff);
            writer.key("contactsDiff");
            writer.value(contactsDiff);
            writer.key("tagsDiff");
            writer.value(tagsDiff);
            writer.key("sharedDiff");
            writer.value(sharedDiff);
            writer.endObject();
        }
    }
}
