package ru.yandex.ps.webtools.aceventura;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;

import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaPrefixParser;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.test.util.JsonChecker;

public class FullCheckIndexHandler implements ProxyRequestHandler {
    private final AceventuraProject project;

    public FullCheckIndexHandler(
        final AceventuraProject project)
    {
        this.project = project;
    }

    protected void database(
        final Context context,
        final FutureCallback<UserData> callback)
        throws BadRequestException
    {
        ProxySession session = context.session();
        AceVenturaPrefix user = context.user();

        MultiFutureCallback<JsonObject> mfcb =
            new MultiFutureCallback<>(
                new DatabaseCallback(context, callback));

        AsyncClient msalClient = project.msalClient();
        QueryConstructor qc = new QueryConstructor("/get-user-contacts?&scope=contacts");
        qc.append("uid", context.user().uid());
        qc.append("userType", context.user().userType().lowName());

        msalClient.execute(
            project.msalHost(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(msalClient),
            mfcb.newCallback());

//        QueryConstructor qc = new QueryConstructor("/get-shared-lists?&scope=contacts");
//        qc.append("uid", user.uid());
//        qc.append("userType", user.userType().lowName());

//        msalClient.execute(
//            project.msalHost(),
//            new BasicAsyncRequestProducerGenerator(qc.toString()),
//            JsonAsyncTypesafeDomConsumerFactory.OK,
//            session.listener().createContextGeneratorFor(msalClient),
//            mfcb.newCallback());

        qc = new QueryConstructor("/get-user-emails?&scope=contacts");
        qc.append("uid", user.uid());
        qc.append("userType", user.userType().lowName());

        msalClient.execute(
            project.msalHost(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(msalClient),
            mfcb.newCallback());


        qc = new QueryConstructor("/get-contacts-list?&scope=contacts");
        qc.append("uid", user.uid());
        qc.append("userType", user.userType().lowName());
        msalClient.execute(
            project.msalHost(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(msalClient),
            mfcb.newCallback());

        qc = new QueryConstructor("/get-shared-owners-lists?&scope=contacts");
        qc.append("uid", user.uid());
        qc.append("userType", user.userType().lowName());
        msalClient.execute(
            project.msalHost(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(msalClient),
            mfcb.newCallback());
        mfcb.done();
    }

    protected void index(
        final Context context,
        final FutureCallback<List<UserData>> callback)
        throws BadRequestException
    {
        ProxySession session = context.session();
        AceVenturaPrefix user = context.user();

        QueryConstructor qc = new QueryConstructor(
            "/search?&service=");
        qc.append("prefix", user.toString());
        qc.append("service", project.defaultService());
        qc.append("text", "av_record_type_p:*");
        qc.append("get", "av_record_type,av_names,av_list_id,av_email_cid,av_email_id,av_email," +
            "av_email_tags,av_contact_tags,av_tags,av_cid,av_list_name," +
            "av_revision,av_shared_owner_uid,av_shared_owner_utype,av_shared_list_id");

        MultiFutureCallback<UserData> mfcb = new MultiFutureCallback<>(callback);

        AsyncClient client = project.searchClient().adjust(session.context());

        List<HttpHost> hosts =
            project.searchmap().searchHosts(new User(project.defaultService(), user));

        for (HttpHost host: hosts) {
            client.execute(
                host,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new IndexCallback(context, host, mfcb.newCallback()));
        }
        mfcb.done();
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        Context context = new Context(session);
        check(context, new Printer(context));
    }

    private void check(
        final Context context,
        final FutureCallback<? super Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>> callback)
        throws HttpException
    {
        DoubleFutureCallback<UserData, List<UserData>> dfcb =
            new DoubleFutureCallback<>(
                new CheckCallback(context, callback));

        database(context, dfcb.first());
        index(context, dfcb.second());
    }

    private static class Printer
        extends AbstractProxySessionCallback<Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>>
    {
        private final Context context;

        public Printer(final Context context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>> map) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
                writer.startObject();
                writer.key("current");
                writer.value(map.get(context.user()));
                writer.key("shared");
                writer.startArray();
                for (Map.Entry<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>> entry: map.entrySet()) {
                    if (entry.getKey().equals(context.user)) {
                        continue;
                    }
                    writer.startObject();
                    writer.key(entry.getKey().toString());
                    writer.value(entry.getValue());
                    writer.endObject();
                }
                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                session.logger().info("Bad json " + sbw.toString());
                failed(ioe);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new StringEntity(sbw.toString(), StandardCharsets.UTF_8));
        }
    }

    private static class Context {
        private final AceVenturaPrefix user;
        private final boolean checkShared;
        private final boolean checkAllReplicas;
        private final ProxySession session;

        public Context(final ProxySession session) throws BadRequestException {
            Long uid = session.params().getLong("uid", -1L);
            if (uid < 0) {
                user = session.params().get("user", new AceVenturaPrefixParser());
            } else {
                user = new AceVenturaPrefix(uid, UserType.PASSPORT_USER);
            }

            this.session = session;
            checkShared = session.params().getBoolean("shared", true);
            checkAllReplicas = session.params().getBoolean("all_replicas", true);
        }

        public Context(
            final ProxySession session,
            final AceVenturaPrefix user,
            final boolean checkShared,
            final boolean checkAllReplicas)
        {
            this.user = user;
            this.checkShared = checkShared;
            this.checkAllReplicas = checkAllReplicas;
            this.session = session;
        }

        public ProxySession session() {
            return session;
        }

        public AceVenturaPrefix user() {
            return user;
        }

        public boolean checkShared() {
            return checkShared;
        }

        public boolean checkAllReplicas() {
            return checkAllReplicas;
        }
    }

    private final class CheckCallback
        extends AbstractFilterFutureCallback<Map.Entry<UserData, List<UserData>>,
        Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>>
    {
        private final Context context;

        public CheckCallback(
            final Context context,
            final FutureCallback<? super Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>> callback)
        {
            super(callback);
            this.context = context;
        }

        private JsonMap formDiff(final Object index, final Object db) {
            JsonMap diffItem = new JsonMap(BasicContainerFactory.INSTANCE);
            diffItem.put("index", new JsonString(String.valueOf(index)));
            diffItem.put("db", new JsonString(String.valueOf(db)));
            return diffItem;
        }

        private List<JsonObject> checkEmails(
            final UserData database,
            final UserData index)
            throws JsonException
        {
            List<JsonObject> diffs = new ArrayList<>();
            for (Map.Entry<String, JsonMap> entry: database.emails().entrySet()) {
                JsonMap dbOne = entry.getValue();
                JsonMap indexOne = index.emails().get(entry.getKey());

                JsonMap diffItem = new JsonMap(BasicContainerFactory.INSTANCE);
                diffItem.put("record_type", new JsonString("email"));
                diffItem.put("id", new JsonString(entry.getKey()));
                if (indexOne == null) {
                    diffItem.put("type", new JsonString("missing"));
                    diffItem.put("data", dbOne);
                    diffs.add(diffItem);
                    continue;
                }

                diffItem.put("type", new JsonString("mismatch"));

                JsonList diff = new JsonList(BasicContainerFactory.INSTANCE);

                long dbRevision = dbOne.getLong("revision");
                long indexRevsion = indexOne.getLong("av_revision", -1L);
                if (dbRevision != indexRevsion) {
                    diff.add(formDiff(indexRevsion, dbRevision));
                }

                String dbEmail = dbOne.getString("email");
                String indexEmail = indexOne.getString("email", null);
                if (!dbEmail.equalsIgnoreCase(indexEmail)) {
                    diff.add(formDiff(indexEmail, dbEmail));
                }
            }

            return diffs;
        }

        private List<JsonObject> checkShared(
            final UserData database,
            final UserData index)
            throws JsonException
        {
            List<JsonObject> diffs = new ArrayList<>();
            for (Map.Entry<AceVenturaPrefix, JsonMap> entry: database.shared().entrySet()) {
                JsonMap dbOne = entry.getValue();
                JsonMap indexOne = index.shared().get(entry.getKey());

                JsonMap diffItem = new JsonMap(BasicContainerFactory.INSTANCE);
                diffItem.put("record_type", new JsonString("shared"));
                diffItem.put("id", new JsonString(entry.getKey().toStringFast()));

                if (indexOne == null) {
                    diffItem.put("type", new JsonString("missing"));
                    diffItem.put("data", dbOne);
                    diffs.add(diffItem);
                    continue;
                }
            }

            return diffs;
        }

        private List<JsonObject> checkContacts(
            final UserData database,
            final UserData index)
            throws JsonException
        {
            List<JsonObject> diffs = new ArrayList<>();
            for (Map.Entry<String, JsonMap> entry: database.contacts().entrySet()) {
                JsonMap dbOne = entry.getValue();
                JsonMap indexOne = index.contacts().get(entry.getKey());

                JsonMap diffItem = new JsonMap(BasicContainerFactory.INSTANCE);
                diffItem.put("record_type", new JsonString("contact"));
                diffItem.put("id", new JsonString(entry.getKey()));

                if (indexOne == null) {
                    diffItem.put("type", new JsonString("missing"));
                    diffItem.put("data", dbOne);
                    diffs.add(diffItem);
                    continue;
                }

                diffItem.put("type", new JsonString("mismatch"));
                JsonList diff = new JsonList(BasicContainerFactory.INSTANCE);

                long dbRevision = dbOne.getLong("revision");
                long indexRevsion = indexOne.getLong("av_revision", -1L);
                if (dbRevision != indexRevsion) {
                    diff.add(formDiff(indexRevsion, dbRevision));
                }

                long dbListId = dbOne.getLong("list_id");
                long indexListId = indexOne.getLong("av_list_id", -1L);
                if (dbListId != indexListId) {
                    diff.add(formDiff(indexListId, dbListId));
                }


                String dbVcard = dbOne.getString("vcard");
                String indexVcard = indexOne.getString("av_vcard", "{}");
                String vcardDiff = new JsonChecker(dbVcard).check(indexVcard);
                if (vcardDiff != null && !vcardDiff.isEmpty()) {
                    diff.add(formDiff(vcardDiff, dbVcard));
                }

                if (!diff.isEmpty()) {
                    diffItem.put("diff", diff);
                }
            }

            return diffs;
        }

        private List<JsonObject> checkHost(
            final UserData database,
            final UserData index)
            throws JsonException
        {
            List<JsonObject> diffs = new ArrayList<>();
            diffs.addAll(checkContacts(database, index));
            diffs.addAll(checkEmails(database, index));
            diffs.addAll(checkShared(database, index));

            return diffs;
        }


        @Override
        public void completed(final Map.Entry<UserData, List<UserData>> entry) {
            ProxySession session = context.session();
            session.logger().info("Check gathering data completed " + context.user());
            UserData database = entry.getKey();
            List<UserData> indexs = entry.getValue();

            Map<HttpHost, List<JsonObject>> diffs = new LinkedHashMap<>();
            try {
                for (UserData index: indexs) {
                    diffs.put(index.host, checkHost(database, index));
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            if (context.checkShared() && !database.shared().isEmpty()) {
                context.session().logger().info("Checking shared " + database.shared().size());
                MultiFutureCallback<Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>> mfcb
                    = new MultiFutureCallback<>(new MergeCallback(callback));
                mfcb.newCallback().completed(Collections.singletonMap(context.user, diffs));
                try {
                    for (AceVenturaPrefix prefix: database.shared.keySet()) {
                        Context subContext =
                            new Context(
                                context.session(),
                                prefix,
                                false,
                                context.checkAllReplicas());
                        check(subContext, mfcb.newCallback());
                    }
                } catch (HttpException e) {
                    failed(e);
                    return;
                }

                mfcb.done();
                return;
            }

            callback.completed(Collections.singletonMap(context.user, diffs));
        }
    }

    private static class MergeCallback
        extends AbstractFilterFutureCallback<List<Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>>,
        Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>>
    {
        public MergeCallback(
            final FutureCallback<? super Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>> callback)
        {
            super(callback);
        }

        @Override
        public void completed(final List<Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>>> maps) {
            Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>> result = new LinkedHashMap<>();
            for (Map<AceVenturaPrefix, Map<HttpHost, List<JsonObject>>> map: maps) {
                result.putAll(map);
            }

            callback.completed(result);
        }
    }
    private static class UserData {
        private final Map<String, JsonMap> contacts;
        private final Map<String, JsonMap> emails;
        private final Map<AceVenturaPrefix, JsonMap> shared;
        private final HttpHost host;

        public UserData(
            final Map<String, JsonMap> contacts,
            final Map<String, JsonMap> emails,
            final Map<AceVenturaPrefix, JsonMap> shared)
        {
            this.contacts = contacts;
            this.emails = emails;
            this.shared = shared;
            this.host = null;
        }

        public UserData(
            final HttpHost host,
            final Map<String, JsonMap> contacts,
            final Map<String, JsonMap> emails,
            final Map<AceVenturaPrefix, JsonMap> shared)
        {
            this.contacts = contacts;
            this.emails = emails;
            this.shared = shared;
            this.host = host;
        }

        public Map<String, JsonMap> contacts() {
            return contacts;
        }

        public Map<String, JsonMap> emails() {
            return emails;
        }

        public Map<AceVenturaPrefix, JsonMap> shared() {
            return shared;
        }
    }

    private static class DatabaseCallback
        extends AbstractFilterFutureCallback<List<JsonObject>, UserData>
    {
        private final Context context;

        public DatabaseCallback(
            final Context context,
            final FutureCallback<? super UserData> callback)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final List<JsonObject> jsonObjects) {
            ProxySession session = context.session();
            try {
                session.logger().info("database completed, " + context.user() + " " +  jsonObjects.size());
                Map<String, JsonMap> contacts = new LinkedHashMap<>();
                for (JsonObject contactObj: jsonObjects.get(0).asList()) {
                    JsonMap map = contactObj.asMap();
                    contacts.put(map.getString("contact_id"), map);
                }

                Map<String, JsonMap> emails = new LinkedHashMap<>();
                for (JsonObject emailObj: jsonObjects.get(1).asList()) {
                    JsonMap map = emailObj.asMap();
                    emails.put(map.getString("email_id"), map);
                }

                Map<AceVenturaPrefix, JsonMap> shared = new LinkedHashMap<>();
                for (JsonObject sharedObj: jsonObjects.get(3).asList()) {
                    JsonMap map = sharedObj.asMap();
                    shared.put(
                        new AceVenturaPrefix(
                            map.getLong("owner_user_id"),
                            map.getEnum(UserType.class, "owner_user_type")),
                        map);
                }

                callback.completed(new UserData(contacts, emails, shared));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private static class IndexCallback
        extends AbstractFilterFutureCallback<JsonObject, UserData>
    {
        private final Context context;
        private final HttpHost host;

        public IndexCallback(
            final Context context,
            final HttpHost host,
            final FutureCallback<? super UserData> callback)
        {
            super(callback);
            this.context = context;
            this.host = host;
        }

        @Override
        public void completed(final JsonObject indexResponse) {
            ProxySession session = context.session();
            session.logger().info("index completed " + host + " " + context.user());
            try {
                Map<String, JsonMap> contacts = new LinkedHashMap<>();
                Map<String, JsonMap> emails = new LinkedHashMap<>();
                Map<AceVenturaPrefix, JsonMap> shared = new LinkedHashMap<>();
                for (JsonObject emailObj: indexResponse.asMap().getList("hitsArray")) {
                    JsonMap map = emailObj.asMap();
                    String recordType = map.getString("av_record_type");
                    if (recordType.equalsIgnoreCase("contact")) {
                        contacts.put(map.getString("av_cid"), map);
                    } else if (recordType.equalsIgnoreCase("email")) {
                        emails.put(map.getString("av_email_id"), map);
                    } else if (recordType.equalsIgnoreCase("shared")) {
                        shared.put(
                            new AceVenturaPrefix(
                                map.getLong("av_shared_owner_uid"),
                                map.getEnum(UserType.class, "av_shared_owner_utype")),
                            map);
                    }
                }

                callback.completed(new UserData(host, contacts, emails, shared));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }
}
