package ru.yandex.ps.webtools.common.peach;

import java.io.IOException;
import java.util.AbstractMap;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.config.check.peach.ImmutablePeachMetricConfig;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.mail.search.web.health.metrics.PeachQueueMetricFactory;
import ru.yandex.mail.search.web.health.update.MetricUpdateTaskFactory;

public abstract class AbstractPeachHandler implements ProxyRequestHandler {
    protected final DefaultPsProject project;
    protected final HealthCheckService health;
    protected final Map<String, Map<String, ImmutablePeachMetricConfig>> configs;

    public AbstractPeachHandler(
        final DefaultPsProject project,
        final HealthCheckService health)
    {
        this.project = project;
        this.health = health;

        Map<String, Map<String, ImmutablePeachMetricConfig>> configs = new LinkedHashMap<>();
        for (MetricUpdateTaskFactory factory: project.config().metrics()) {
            if (factory instanceof PeachQueueMetricFactory) {
                ImmutablePeachMetricConfig config = ((PeachQueueMetricFactory) factory).config();
                //TODO: check it into config
                //if (configs.get(config.name()) != null ) {
                //    throw new BadRequestException("Duplicate peach config name found");
                //}
                Map<String, ImmutablePeachMetricConfig> queueMap = new LinkedHashMap<>();
                for (String queue: config.queues()) {
                    queueMap.put(queue, config);
                }

                configs.put(config.name(), Collections.unmodifiableMap(queueMap));
            }
        }

        this.configs = Collections.unmodifiableMap(configs);
    }

    protected Set<String> services(
        final ProxySession session,
        final ImmutablePeachMetricConfig config,
        final String queueName)
        throws HttpException, IOException
    {
        Collection<String> services = config.services();
        if (services.isEmpty()) {
            services = Collections.singleton(project.defaultService());
        }

        return new LinkedHashSet<>(services);
    }

    protected ImmutablePeachMetricConfig getPeachConfig(
        final PeachRequestContext context)
        throws BadRequestException
    {

        Map<String, ImmutablePeachMetricConfig> queueMap = configs.get(context.configName());
        if (queueMap == null) {
            throw new BadRequestException("No config found with name " + context.configName());
        }

        ImmutablePeachMetricConfig config = queueMap.get(context.queueName());

        if (config == null) {
            throw new BadRequestException("No config found with queue name " + context.queueName());
        }

        return config;
    }

    protected Map.Entry<ShardReplica, ImmutablePeachMetricConfig> getReplicaByShard(
        final PeachRequestContext context)
        throws HttpException, IOException
    {
        Set<ShardReplica> replicas = new LinkedHashSet<>();

        ImmutablePeachMetricConfig config = getPeachConfig(context);

        for (String service: services(context.session(), config, context.queueName())) {
            ProjectQueue queue = health.projectRoot().queue(service);
            if (queue == null) {
                continue;
            }

            Shard shard = queue.shard(context.shard());
            if (shard == null) {
                continue;
            }

            ShardReplica replica = shard.replica(context.hostname());
            if (replica == null) {
                continue;
            }

            replicas.add(replica);
        }

        if (replicas.size() != 1) {
            throw new BadRequestException(
                "No replicas or more than 1 found for config "
                    + " queues " + services(context.session(), config, context.queueName())
                    + " host " + context.hostname());
        }

        ShardReplica replica = replicas.iterator().next();
        return new AbstractMap.SimpleEntry<>(replica, config);
    }
}
