package ru.yandex.ps.webtools.common.peach;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.config.check.peach.ImmutablePeachMetricConfig;
import ru.yandex.mail.search.web.config.check.peach.PeachMetricConfigDefaults;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.ShardGroup;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.mail.search.web.searchmap.MailSearchHost;

public abstract class AbstractValidatePeachRecordsHandler extends AbstractPeachHandler {
    public AbstractValidatePeachRecordsHandler(
        final DefaultPsProject project,
        final HealthCheckService health)
    {
        super(project, health);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        PeachRequestContext context = new PeachRequestContext(project, session);

        if (context.shard() >= 0) {
            Map.Entry<ShardReplica, ImmutablePeachMetricConfig> entry = getReplicaByShard(context);
            FindAllInvalidConsumer consumer =
                new FindAllInvalidConsumer(session, context.shard(), new SingleShardValidationResultPrinter(context));

            ShardReplica replica = entry.getKey();
            Context searchContext =
                new Context(
                    context,
                    new HttpHost(replica.host().hostname(), replica.host().searchPort()),
                    (long) context.shard(),
                    entry.getValue(),
                    consumer);

            executeNext(searchContext, new SearchCallback(session, searchContext));
        } else {
            // fetch shards
            ImmutablePeachMetricConfig config = getPeachConfig(context);

            ShardGroup group = null;
            for (String service: services(context.session(), config, context.queueName())) {
                ProjectQueue queue = health.projectRoot().queue(service);
                if (queue == null) {
                    continue;
                }

                List<ShardGroup> groups = queue.shardsByHost(context.hostname());
                if (groups.size() == 0) {
                    throw new BadRequestException("No shard groups for hostname " + context.hostname());
                }

                if (groups.size() > 1) {
                    throw new BadRequestException("More than 1 shard group for hostname " + context.hostname());
                }

                group = groups.get(0);
            }

            if (group == null) {
                throw new BadRequestException("No shard group for hostname " + context.hostname());
            }

            MailSearchHost host = null;
            for (MailSearchHost h: group.node().hosts()) {
                if (context.hostname().equalsIgnoreCase(h.hostname())) {
                    host = h;
                    break;
                }
            }

            if (host == null) {
                throw new ServerException(
                    HttpStatus.SC_INTERNAL_SERVER_ERROR,
                    "Missing host in group " + context.hostname() + ' ' + group.node());
            }

            StringBuilder fetchShardsUri = new StringBuilder(config.queuelenBaseUri().toString());
            fetchShardsUri.append("peach_url:*");
            context.client().execute(
                new HttpHost(host.hostname(), host.searchPort()),
                new BasicAsyncRequestProducerGenerator(fetchShardsUri.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(context.client()),
                new ShardListCallback(context, config, group, host));
        }

    }

    protected void executeNext(final Context context, final SearchCallback callback) {
        context.client().execute(
            context.host(),
            new BasicAsyncRequestProducerGenerator(context.searchUri()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session.listener().createContextGeneratorFor(context.client()),
            callback);
    }

    /**
     * @param consumer
     * @param task
     * @return true - if should continue, falase if break
     */
    protected abstract boolean validateRecord(
        final PeachRecordCheckConsumer consumer,
        final JsonMap task);

    private final class ShardListCallback extends AbstractProxySessionCallback<JsonObject> {
        private final PeachRequestContext context;
        private final ShardGroup group;
        private final MailSearchHost backend;
        private final ImmutablePeachMetricConfig config;

        public ShardListCallback(
            final PeachRequestContext context,
            final ImmutablePeachMetricConfig config,
            final ShardGroup group,
            final MailSearchHost host)
        {
            super(context.session());
            this.context = context;
            this.group = group;
            this.backend = host;
            this.config = config;
        }

        @Override
        public void completed(final JsonObject response) {
            Set<Long> shards = Collections.emptySet();
            try {
                Set<String> keys = response.asMap().keySet();
                shards = new LinkedHashSet<>(keys.size());
                for (String key: keys) {
                    String[] split = key.split("#");
                    if (split.length != 2) {
                        failed(
                            new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR,
                                "Invalid printkeys data " + JsonType.NORMAL.toString(response)));
                        return;
                    }

                    shards.add(Long.parseLong(split[0]));
                }
            } catch (JsonException | NumberFormatException e) {
                failed(
                    new ServerException(
                        HttpStatus.SC_INTERNAL_SERVER_ERROR,
                        "Bad printkeys response "  + JsonType.NORMAL.toString(response)));
            }

            try {
                HttpHost host = new HttpHost(backend.hostname(), backend.searchPort());
                context.session().logger().info("Shard list completed with " + shards);

                MultiFutureCallback<PeachShardValidateReport> mfcb
                    = new MultiFutureCallback<>(new MultiShardValidationPrinter(context));

                for (Long shard: shards) {
                    FindAllInvalidConsumer consumer = new FindAllInvalidConsumer(session, shard, mfcb.newCallback());
                    Context searchContext = new Context(context, host, shard, config, consumer);
                    executeNext(searchContext, new SearchCallback(session, searchContext));
                }
                mfcb.done();
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private final class SearchCallback
        extends AbstractProxySessionCallback<JsonObject> {
        private final PeachRecordCheckConsumer consumer;
        private final Context context;

        public SearchCallback(
            final ProxySession session,
            final Context context) {
            super(session);
            this.consumer = context.consumer();
            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonList hits =
                    resultObj.asMap().getList("hitsArray");
                for (JsonObject hitObj : hits) {
                    JsonMap task = hitObj.asMap();

                    // We need to check by field persistence
                    if (!validateRecord(consumer, task)) {
                        return;
                    } else {
                        continue;
                    }
                }

                int position = hits.size();
                context.currentPosition.set(position);
                if (hits.size() < context.batchSize || position >= context.maxLength) {
                    consumer.exhausted();
                    return;
                }

                executeNext(context, this);
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    protected class Context {
        private final PeachRequestContext requestContext;
        private final ProxySession session;
        private final int maxLength;
        private final int batchSize;
        private final String searchUri;
        private final HttpHost searchHost;
        private final PeachRecordCheckConsumer consumer;
        private final AtomicInteger currentPosition
            = new AtomicInteger();

        private final ImmutablePeachMetricConfig config;
        private final String queueName;

        public Context(
            final PeachRequestContext requestContext,
            final HttpHost searchHost,
            final Long shardId,
            final ImmutablePeachMetricConfig config,
            final PeachRecordCheckConsumer consumer)
            throws BadRequestException
        {
            this.requestContext = requestContext;
            this.consumer = consumer;
            this.config = config;
            this.session = requestContext.session();
            this.maxLength = session.params().getInt("max_length", 5000);
            //this.batchSize = session.params().getInt("batch_size", 100);
            this.batchSize = maxLength;
            this.queueName = requestContext.queueName();
            // Get peach config - how we need check for name

            // TODO: Default peach config creates by metric name = peach
            //  In default config we doesn't have additional params, so check it here

            if (config.sortField() == null) {
                throw new BadRequestException(
                    "You must specify peach sort-field in config for this action.");
            }

            this.searchHost = searchHost;

            StringBuilder sb = new StringBuilder(config.validateBaseUri().toString());

            //TODO: How we can iterate over queues without name?
            // This must be forbidden
            // For mail_search_prod we have only one queue.

            if (queueName.equals(PeachMetricConfigDefaults.DEFAULT_PEACH_QUEUE)) {
                sb.append(this.config.noQueueFilterQuery());
            } else {
                sb.append(this.config.queueFilterQuery());
                sb.append(queueName);
            }
            sb.append("&get=");
            sb.append(String.join(",", this.config.fields()));
            sb.append("&sort=");
            sb.append(this.config.sortField());
            sb.append("&prefix=");
            sb.append(shardId);

            this.searchUri = searchHost.toString() + sb.toString();
        }

        public String searchUri() {
            StringBuilder sb = new StringBuilder(searchUri);
            sb.append("&length=");
            sb.append(currentPosition.get() + batchSize);
            return sb.toString();
        }

        public HttpHost host() {
            return searchHost;
        }

        public PeachRecordCheckConsumer consumer() {
            return consumer;
        }

        public AsyncClient client() {
            return requestContext.client();
        }
    }

}
