package ru.yandex.ps.webtools.common.peach;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.config.check.peach.ImmutablePeachMetricConfig;
import ru.yandex.mail.search.web.config.check.peach.PeachMetricConfigDefaults;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.parser.string.PositiveIntegerValidator;

public class DropPeachRecordsHandler extends AbstractPeachHandler {
    private static final String OR = "+OR+";

    public DropPeachRecordsHandler(
        final DefaultPsProject project,
        final HealthCheckService health)
    {
        super(project, health);
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        PeachRequestContext context = new PeachRequestContext(project, session);
        if (context.shard() < 0) {
            throw new BadRequestException("Invalid shard id " + context.shard());
        }

        Map.Entry<ShardReplica, ImmutablePeachMetricConfig> entry = getReplicaByShard(context);
        ImmutablePeachMetricConfig config = entry.getValue();
        ShardReplica replica = entry.getKey();

        List<String> ranges = session.params().getAll("range");
        //by ranges
        if (ranges.isEmpty()) {
            throw new BadRequestException("No ranges supplied");
        }

        if (config.sortField() == null) {
            throw new BadRequestException("You must specify peach sort-field in config for this action.");
        }

        StringBuilder sb = new StringBuilder("/delete?peach&webtools&prefix=");
        sb.append(replica.shard().id());

        sb.append("&text=peach_url:1+AND+");

        //TODO: How we can iterate over queues without name?
        // This must be forbidden
        // For mail_search_prod we have only one queue.

        if (context.queueName().equals(PeachMetricConfigDefaults.DEFAULT_PEACH_QUEUE)) {
            sb.append("(");
        } else {
            sb.append("peach_queue:");
            sb.append(context.queueName());
            sb.append("+AND+");
            sb.append('(');
        }

        for (String range: ranges) {
            String[] split = range.split("-");
            int start;
            int end;

            try {
                start = PositiveIntegerValidator.INSTANCE.apply(split[0].trim());
                end = PositiveIntegerValidator.INSTANCE.apply(split[1].trim());
            } catch (Exception e) {
                throw new BadRequestException("Failed to parse range " + range);
            }

            if (start > end) {
                throw new BadRequestException("Invalid range " + range);
            }
            sb.append(config.sortField());
            sb.append(":[");
            sb.append(start);
            sb.append("+TO+");
            sb.append(end);
            sb.append(']');
            sb.append(OR);
        }
        // remove last OR
        sb.setLength(sb.length() - OR.length());
        sb.append(')');

        session.logger().info("Delete uri " + sb.toString());
        AsyncClient client = project.searchClient().adjust(session.context());
        client.execute(
            new HttpHost(replica.host().hostname(),replica.host().indexerPort()),
            new BasicAsyncRequestProducerGenerator(sb.toString()),
            EmptyAsyncConsumerFactory.ANY_GOOD,
            session.listener().createContextGeneratorFor(client),
            new DeleteCallback(session));

    }

    private static class DeleteCallback extends AbstractProxySessionCallback<Object> {
        public DeleteCallback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Object o) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
