package ru.yandex.ps.webtools.mail;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.http.HttpException;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.mail.search.web.health.HealthInfoHandler;
import ru.yandex.mail.search.web.health.base.ProjectQueue;
import ru.yandex.mail.search.web.health.base.Shard;
import ru.yandex.mail.search.web.health.base.ShardReplica;
import ru.yandex.mail.search.web.info.ExtractChain;
import ru.yandex.mail.search.web.info.InfoExtractorFactory;
import ru.yandex.mail.search.web.info.PatternExtractChain;
import ru.yandex.parser.searchmap.SearchMap;

public class MailHealthInfoHandler extends HealthInfoHandler {
    private static final ExtractChain UID_EXTRACT_CHAIN =
        new PatternExtractChain(
            InfoExtractorFactory.ONE_LONG,
            InfoExtractorFactory.UID_EXTRACT_CHAIN);

    private final MailSearchProject mailSearch;
    private final HealthCheckService service;

    public MailHealthInfoHandler(
        final MailSearchProject mailSearch,
        final HealthCheckService service)
    {
        super(service);

        this.mailSearch = mailSearch;
        this.service = service;
    }

    protected void handleByUid(
        final ProxySession session,
        final Long uid)
        throws HttpException, IOException
    {
        final Map<String, Map<Integer, List<ShardReplica>>> result =
            new LinkedHashMap<>();

        if (BlackboxUserinfo.corp(uid)) {
            if (!mailSearch.config().projectId().contains("corp")) {
                throw new BadRequestException(uid.toString() + " is corp user");
            }
        } else {
            if (mailSearch.config().projectId().contains("corp")) {
                throw new BadRequestException(
                    uid.toString() + " is not corp user");
            }
        }

        int shardId = (int) (uid % SearchMap.SHARDS_COUNT);
        for (ProjectQueue queue: service.projectRoot().queues().values()) {
            Shard shard = queue.shard(shardId);
            if (shard != null) {
                for (ShardReplica replica: shard.replicas().values()) {
                    result.computeIfAbsent(
                        replica.host().hostname(),
                        (h) -> new LinkedHashMap<>())
                        .computeIfAbsent(shard.id(),
                            (k) -> new ArrayList<>())
                        .add(replica);
                }
            }
        }

        outputOneShard(session, result);
    }

//    protected void handleByHostPrefix(
//        final ProxySession session,
//        final String hostName)
//        throws HttpException, IOException
//    {
//        final Map<String, List<ShardReplica>> result = new LinkedHashMap<>();
//        for (ProjectQueue queue: service.projectRoot().queues().values()) {
//            for (ShardGroup shardGroup: queue.groups()) {
//                for (MailSearchHost host: shardGroup.node().hosts()) {
//                    if (host.hostname().startsWith(hostName)) {
//                        for (Shard shard: shardGroup.shards()) {
//                            result.computeIfAbsent(
//                                host.hostname(),
//                                (h) -> new ArrayList<>())
//                                .add(shard.replica(host.hostname()));
//                        }
//                    }
//                }
//            }
//
//            Shard shard = queue.shard(shardId);
//            if (shard != null) {
//                for (ShardReplica replica: shard.replicas().values()) {
//                    if (!hostFilter.contains(replica.host().hostname())) {
//                        continue;
//                    }
//                    result.computeIfAbsent(
//                        replica.host().hostname(),
//                        (h) -> new ArrayList<>()).add(replica);
//                }
//            }
//        }
//    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        String request = session.params().getString("request");
        if (request.isEmpty()) {
            handleEmpty(session, service.projectRoot().queues().values());
            return;
        }

        try {
            Long longValue = Long.parseLong(request);
            if (longValue < SearchMap.SHARDS_COUNT) {
                handleByShardId(
                    session,
                    service.projectRoot().queues().values(),
                    longValue.intValue());
                return;
            }
        } catch (NumberFormatException nfe) {
        }

        // try uid
        String uidStr = UID_EXTRACT_CHAIN.extract(request);
        if (uidStr != null) {
            Long uid = Long.parseLong(uidStr);

            handleByUid(session, uid);
            return;
        }

        handleByHost(
            session,
            service.projectRoot().queues().values(),
            request.trim().toLowerCase(Locale.ENGLISH));
        // suppose it is host
    }
}
