package ru.yandex.ps.webtools.mail;

import java.io.IOException;
import java.net.ConnectException;
import java.net.URISyntaxException;
import java.util.function.Function;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.MailReindexHandler;
import ru.yandex.mail.search.web.WebApi;
import ru.yandex.mail.search.web.config.mail.ImmutableMailSearchConfig;
import ru.yandex.mail.search.web.health.base.PsProjectType;
import ru.yandex.mail.search.web.images.ImagesHandler;
import ru.yandex.mail.search.web.info.InfoHandler;
import ru.yandex.mail.search.web.mail.checkindex.CheckIndexHandler;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.ps.webtools.common.peach.DropPeachRecordsHandler;
import ru.yandex.ps.webtools.mail.peach.MailPeachValidateRecordsHandler;

public class MailSearchProject extends DefaultPsProject {
    private static final Function<Exception, RequestErrorType>
        ERROR_CLASSIFIER
        = (e) -> {
        if (e instanceof ConnectException) {
            return RequestErrorType.HOST_NON_RETRIABLE;
        }

        return RequestErrorType.ERROR_CLASSIFIER.apply(e);
    };

    private final ImmutableMailSearchConfig config;

    private final AsyncClient msearchProxyClient;
    private final AsyncClient msalClient;
    private final AsyncClient searchClient;
    private final AsyncClient filterSearchClient;
    private final AsyncClient bpReindexProducer;
    private final AsyncClient iexClient;

    private final Tvm2TicketRenewalTask filterSearchTvm2Task;

    public MailSearchProject(
        final WebApi webApi,
        final PsProjectType project,
        final ImmutableMailSearchConfig config)
        throws IOException
    {
        super(webApi, project, config);

        this.config = config;

        this.msearchProxyClient =
            webApi.client(
                config.projectId() + "-MsearchProxy",
                config.mailSearch());
        this.msalClient =
            webApi.client(config.projectId() + "-Msal", config.msal());
        this.bpReindexProducer =
            webApi.client(
                config.projectId() + "-BpReindexProducer",
                config.reindexProducer());

        this.iexClient = webApi.client(config.projectId() + "IEX", config.iex());
        this.filterSearchClient =
            webApi.client(config.projectId() + "FilterSearch", config.filterSearch());

        searchClient =
            webApi.client(
                config.projectId() + "-SearchClient",
                config.searchConfig(),
                ERROR_CLASSIFIER);

        try {
            filterSearchTvm2Task = new Tvm2TicketRenewalTask(
                logger.addPrefix("filterSearchTvm2"),
                webApi.serviceContextRenewalTask(),
                config.filterSearchTvm2());
        } catch (HttpException | JsonException | URISyntaxException e) {
            throw new IOException(e);
        }

        // handlers
        webApi.register(
            config,
            "/info",
            new InfoHandler(this),
            RequestHandlerMapper.GET);

        webApi.register(
            config,
            "/reindex",
            new MailReindexHandler(this),
            RequestHandlerMapper.GET);

        webApi.register(
            config,
            "/checkindex",
            new CheckIndexHandler(this),
            RequestHandlerMapper.GET);

        webApi.register(
            config,
            "/newsImages",
            new ImagesHandler(this),
            RequestHandlerMapper.GET);

        webApi.register(
            config,
            "/peach-validate",
            new MailPeachValidateRecordsHandler(this, healthCheckService));
        webApi.register(config, "/peach-drop", new DropPeachRecordsHandler(this, healthCheckService));
    }

    @Override
    public void start() {
        filterSearchTvm2Task.start();
        super.start();
    }

    @Override
    public void close() throws IOException {
        filterSearchTvm2Task.cancel();
        super.close();
    }

    public AsyncClient reindexer(final long uid) {
        return bpReindexProducer;
    }

    public HttpHost reindexerHost(final long uid) {
        return config.reindexProducer().host();
    }

    public AsyncClient filterSearchClient() {
        return filterSearchClient;
    }

    public ImmutableMailSearchConfig config() {
        return config;
    }

    public SearchMap searchmap() {
        return searchmap;
    }

    public AsyncClient msearchProxyClient() {
        return msearchProxyClient;
    }

    public AsyncClient msalClient() {
        return msalClient;
    }

    public AsyncClient searchClient() {
        return searchClient;
    }

    public AsyncClient iexClient() {
        return iexClient;
    }

    @Override
    public String defaultService(final String prefix) {
        return config.mainQueue();
    }

    public WebApi webApi() {
        return webApi;
    }

    public String filterSearchTicket() {
        return filterSearchTvm2Task.ticket();
    }

    @Override
    protected ProxyRequestHandler healthInfoHandler() {
        return new MailHealthInfoHandler(this, healthCheckService);
    }
}
