package ru.yandex.ps.webtools.mail.peach;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.search.web.DefaultPsProject;
import ru.yandex.mail.search.web.health.HealthCheckService;
import ru.yandex.ps.webtools.common.peach.AbstractValidatePeachRecordsHandler;
import ru.yandex.ps.webtools.common.peach.PeachRecordCheckConsumer;
import ru.yandex.util.string.UnhexStrings;

public class MailPeachValidateRecordsHandler extends AbstractValidatePeachRecordsHandler {
    public MailPeachValidateRecordsHandler(
        final DefaultPsProject project,
        final HealthCheckService health)
    {
        super(project, health);
    }

    @Override
    protected boolean validateRecord(
        final PeachRecordCheckConsumer consumer,
        final JsonMap task)
    {
        int peachSeq = -1;
        String pk = null;

        String peachUrl;
        String payload;
        try {
            peachSeq = task.getInt("peach_sequence");
            pk = task.getString("url");
            peachUrl = task.getString("peach_url", null);
            payload = task.getString("peach_payload");
        } catch (JsonException je) {
            return consumer.invalid(
                task,
                peachSeq,
                "Invalid task format: "
                    + je.getMessage());
        }

        if (payload == null || peachUrl == null) {
            return consumer.invalid(
                task,
                peachSeq,
                "Payload or peach url is null");
        }

        try {
            byte[] unhexed = UnhexStrings.unhex(payload);
            TypesafeValueContentHandler.parse(
                new String(unhexed, StandardCharsets.UTF_8));
        } catch (Exception e) {
            return consumer.invalid(
                task,
                peachSeq,
                "Unable unhex/parse json: " + e.toString());
        }

        try {
            new URI(peachUrl).parseServerAuthority();
        } catch (URISyntaxException e) {
            return consumer.invalid(
                task,
                peachSeq,
                "Unable parse url bad url detected: " + e.toString());
        }

        return consumer.valid(task, peachSeq);
    }
}
