# -*- coding: utf-8 -*-
import requests
import logging

from django.contrib.auth import login
from django.contrib.auth.models import User
from django.utils.deprecation import MiddlewareMixin

from requests.adapters import HTTPAdapter

from mail_search_webtools import settings
from mail_search_webtools.settings import WEBTOOLS_API_HOST
from webtools.models import LinkedYandexUser

log = logging.getLogger('django')

AUTH_URL = '/session/check?host={host}&sessionid={sessionid}'


class AuthException(Exception):
    """
    Базовый класс для исключений аутентификации.
    """
    pass


class AuthRequired(AuthException):
    """
    Эксепшн бросается в пользовательском коде, чтобы
    инициировать редирект на паспорт в middleware.
    """
    def __init__(self, retpath='', *args, **kwargs):
        self.retpath = retpath
        super(AuthRequired, self).__init__(*args, **kwargs)


class InvalidProtocol(AuthException):
    """
    Для корректной аутентификации в данном случае требуется соединение по HTTPS.
    """
    pass


class MailSearchAdminAuthMiddleware(MiddlewareMixin):
    api_session = None

    def process_request(self, request):
        if hasattr(request, 'user') and not request.user.is_anonymous():
            return

        if not self.api_session:
            self.api_session = requests.Session()
            self.api_session.mount(
                settings.WEBTOOLS_API_HOST,
                HTTPAdapter(max_retries=settings.WEBTOOLS_SESSION_CHECK_RETRIES))

        params = self.extract_params(request)
        log.info('User not passed django normal auth ' + unicode(params))

        if not params.get('session_id') or not params.get('yandex_login') or not params.get('userip'):
            log.info('Not enough data for user auth ' + unicode(params))
            return

        resp = self.api_session.get(
            settings.WEBTOOLS_API_HOST + AUTH_URL.format(
                sessionid=params['session_id'],
                host=params['server_host']),
            headers={'X-Real-IP': params['userip']},
            timeout=30)
        if resp.status_code != 200:
            log.warning('Bad cookie for ' + params['yandex_login'] + ' ' + resp.content)
            return

        log.info('Cookie ok for ' + params['yandex_login'])
        user, created = User.objects.get_or_create(
            username=params['yandex_login'],
            email=params['yandex_login'] + '@yandex-team.ru')

        if created:
            log.info('Created new user ' + params['yandex_login'])
            api_resp = self.api_session.get(
                WEBTOOLS_API_HOST + '/yandex/user/accounts?login=' + params['yandex_login'],
                timeout=30)
            if api_resp.status_code != 200:
                log.warning('Failed to resolve linked accounts ' + resp.status_code + ' ' + resp.content)

            accounts_added = []
            for account in api_resp.json():
                linked_user = LinkedYandexUser(
                    user=user,
                    uid=account['uid'],
                    login=account['login'],
                    email=account['login'] + '@yandex.ru')
                linked_user.save()
                accounts_added.append(unicode(account['login']))

            request.user_created = True
            request.linked_accounts = accounts_added

        user.backend = 'django.contrib.auth.backends.ModelBackend'
        login(request, user)
        log.info('User logged in ' + params['yandex_login'])

    def extract_params(self, request):
        session_id = self._get_session_id(request)
        userip = request.META.get('REMOTE_ADDR')
        if not userip:
            userip = request.META.get('X-Real-IP')

        return dict(
            session_id=session_id,
            yandex_login=self.get_yandex_user_name(request),
            sessionid2=self._get_sessionid2(request),
            server_host=settings.WEBTOOLS_BALANCER,
            userip=userip,
            sessionid2_required=settings.YAUTH_SESSIONID2_REQUIRED,
        )

    def get_yandex_user_name(self, request):
        return request.COOKIES.get('yandex_login')

    def _get_session_id(self, request):
        return request.COOKIES.get('Session_id')

    def _get_sessionid2(self, request):
        if not settings.YAUTH_SESSIONID2_REQUIRED:
            return

        if not request.is_secure():
            log.error('Authorization error: HTTPS connection is required when using "sessionid2"')
            raise InvalidProtocol('Authorization error: HTTPS connection is required')

        sessionid2 = request.COOKIES.get('seddionid2')

        if not sessionid2 and settings.YAUTH_SESSIONID2_REQUIRED:
            log.warning('Authorization error: no sessionid2 cookie, but is mandatory')
            raise AuthException("Not enough data for auth")

        return request.COOKIES.get('sessionid2')

