from datetime import timedelta, datetime

from django import forms
from django.forms import widgets

form_control_attrs = {'class': 'form-control'}
file_form_control_attrs = {'class': 'form-control-file'}

FIELDS = (
    "uid",
    "card_id",
    "ip",
    "as",
    "user_phone",
    "user_agent",
    "card_bin",
    "card_isoa2",
    "currency",
    "device_id",
    "taxi_car_number",
    "taxi_driver_license",
    "card_brand",
    "terminal_id",
    "order_id",
    "order_type",
    "google_token_type",
)

DATETIME_LOCAL_FORMAT = "%Y-%m-%dT%H:%M:%S"

class ChannelForm(forms.Form):
    channel = forms.ChoiceField()

    def __init__(self, *args, **kwargs):
        self.channels = kwargs.pop("channels")
        super(ChannelForm, self).__init__(*args, **kwargs)
        channels_choices = [(choice, choice) for choice in self.channels]
        self.fields['channel'] = forms.ChoiceField(choices=channels_choices,
                                                   widget=widgets.Select(
                                                       attrs={'class': 'form-control',
                                                              'size': '10'}))


class ChannelFormWithAny(forms.Form):
    channel = forms.MultipleChoiceField(required=False)

    def __init__(self, *args, **kwargs):
        self.channels = kwargs.pop("channels")
        super(ChannelFormWithAny, self).__init__(*args, **kwargs)
        channels_choices = [(choice, choice) for choice in self.channels]
        self.fields['channel'] = forms.MultipleChoiceField(choices=channels_choices,
                                                           required=False,
                                                           widget=widgets.SelectMultiple(
                                                               attrs={'class': 'form-control',
                                                                      'size': '10'}))


list_name_attrs = {"list": "lists-names"}
list_name_attrs.update(form_control_attrs)


class NewListForm(ChannelForm):
    field = forms.ChoiceField(choices=tuple((field, field) for field in FIELDS),
                              widget=widgets.Select(attrs=form_control_attrs))
    list_name = forms.CharField(max_length=200, widget=widgets.TextInput(attrs=list_name_attrs))
    days = forms.DurationField(widget=widgets.TextInput(attrs=form_control_attrs))
    reason = forms.CharField(max_length=200, widget=widgets.TextInput(attrs=form_control_attrs))

    text_items = forms.CharField(widget=widgets.Textarea(attrs=form_control_attrs), required=False,
                                 label="'\\s+' separated values")
    file_items = forms.FileField(widget=widgets.FileInput(attrs=file_form_control_attrs),
                                 required=False,
                                 label="File with '\\s+' separated values")


class ListsRequestForm(ChannelForm):
    list_name = forms.CharField(max_length=200, widget=widgets.TextInput(attrs=list_name_attrs))

    value = forms.CharField(max_length=200,
                            widget=widgets.TextInput(attrs=form_control_attrs),
                            required=False)


class TransactionsRequestForm(ChannelFormWithAny):
    query_attrs = {"list": "search-fields"}
    query_attrs.update(form_control_attrs)
    query = forms.CharField(widget=widgets.TextInput(attrs=query_attrs), required=False)
    limit = forms.DurationField(
        widget=widgets.TextInput(
            attrs=form_control_attrs
        ),
        required=False,
        initial=100
    )

    since = forms.CharField(
        widget=widgets.TextInput(attrs={'class': 'form-control'}),
        initial=lambda: (datetime.now() - timedelta(minutes=5)).strftime(DATETIME_LOCAL_FORMAT),
        required=True)

    until = forms.CharField(
        widget=widgets.TextInput(attrs={'class': 'form-control'}),
        initial=lambda: (datetime.now()).strftime(DATETIME_LOCAL_FORMAT),
        required=True)
