import json
import logging
import re
import traceback
import urllib
from datetime import datetime

import requests
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.http import HttpResponseRedirect
from django.shortcuts import render
from django.urls import reverse
from django.views.generic import FormView
from django.views.generic.base import ContextMixin, TemplateView
from django.views.generic.edit import ProcessFormView

import utils
from mail_search_webtools.settings import SO_FRAUD_BACKEND_HOST
from webtools.disk.staff_views import copy_options, get_user_channels
from webtools.so_fraud.forms import NewListForm, ListsRequestForm, TransactionsRequestForm, \
    DATETIME_LOCAL_FORMAT
from webtools.views_utils import default_context

logger = logging.getLogger("so_fraud")


class StaffMemberView(LoginRequiredMixin, UserPassesTestMixin, ContextMixin, ProcessFormView):
    def get_context_data(self, **kwargs):
        options = copy_options()
        for k, v in options.iteritems():
            v['checked'] = True

        context = {
            'options': options.values(),
            'collapsed': True,
            'request_text': self.request.GET.get('request_text', ''),
            'options_js': json.dumps(options.values()),
            'enviroment_default': "&prefix=0&service=so_fraud_data&length=10&get=*",
        }

        context.update(default_context(self.request, self.kwargs['project']))
        context.update(super(StaffMemberView, self).get_context_data(**kwargs))
        return context

    def test_func(self):
        return self.request.user.is_active and self.request.user.is_staff

    def get_login_url(self):
        return 'admin:login'

    def handle_get(self, request, *args, **kwargs):
        return super(StaffMemberView, self).get(request, *args, **kwargs)

    def get(self, request, *args, **kwargs):
        try:
            return self.handle_get(request, *args, **kwargs)
        except Exception as _:
            return render(self.request, 'so_fraud/error_template.html', {
                "error": traceback.format_exc()
            })


class StaffMemberFormView(StaffMemberView, FormView):
    template_name = 'so_fraud/boostrap_form.html'

    def get_initial(self):
        initial = {k: urllib.unquote_plus(value) for k, value in self.request.GET.items() if value}

        channel = initial.get('channel')
        try:
            if channel is not None:
                initial['channel'] = json.loads(channel)
        except Exception as e:
            logger.error("cannot parse json from " + channel + ":" + e.message)

        return initial

    def get_form_kwargs(self):
        kwargs = super(StaffMemberFormView, self).get_form_kwargs()
        kwargs['channels'] = get_user_channels(self.request.user)
        return kwargs

    def handle_form_valid(self, form):
        return super(StaffMemberFormView, self).form_valid(form)

    def form_valid(self, form):
        try:
            return self.handle_form_valid(form)
        except Exception as _:
            return render(self.request, 'so_fraud/error_template.html', {
                "error": traceback.format_exc()
            })


class NewListView(StaffMemberFormView):
    form_class = NewListForm
    template_name = 'so_fraud/lists_update_template.html'

    re_space = re.compile("\\s+")

    def get_context_data(self, **kwargs):
        context = {"lists_names_by_channels": json.dumps(
            utils.get_all_lists_names(self.get_form().channels))}
        context.update(super(NewListView, self).get_context_data(**kwargs))
        return context

    @staticmethod
    def gen_items_batches(form, batch_size):
        batch = []
        for item in NewListView.re_space.split(form.data["text_items"]):
            if item:
                batch.append(item)
                if len(batch) >= batch_size:
                    yield batch
                    batch = []

        for file in form.files.values():
            with file:
                for item in NewListView.re_space.split(str(file.file.read())):
                    if item:
                        batch.append(item)
                        if len(batch) >= batch_size:
                            yield batch
                            batch = []

        if len(batch) > 0:
            yield batch

    def handle_form_valid(self, form):
        now = utils.now_milliseconds()
        milliseconds_per_day = 24 * 60 * 60 * 1000
        results = []

        for batch in self.gen_items_batches(form, 1000):
            data = {"channel": (form.data["channel"].split('/'))[0],
                    "sub_channel": (form.data["channel"].split('/'))[1],
                    "list_name": form.data["list_name"],
                    "field": form.data["field"], "author": self.request.user.username,
                    "from": now,
                    "to": now + int(form.data["days"]) * milliseconds_per_day,
                    "type": "FAST_LIST",
                    "action": "update",
                    "items": batch,
                    "reason": form.data["reason"]
                    }

            response = requests.post(SO_FRAUD_BACKEND_HOST + '/update_list', json=data, timeout=None)
            if response.status_code != 200:
                raise Exception(response.content)

            results.append({
                'status': response.status_code,
                'content': response.content,
            })

        result_context = {
            "results": results
        }
        result_context.update(self.get_context_data())

        return render(self.request, 'so_fraud/list_create_result.html', result_context)


class ListsSearchView(StaffMemberFormView):
    form_class = ListsRequestForm
    template_name = 'so_fraud/lists_template.html'

    def get_context_data(self, **kwargs):
        context = {"lists_names_by_channels": json.dumps(
            utils.get_all_lists_names(self.get_form().channels))}
        context.update(super(ListsSearchView, self).get_context_data(**kwargs))
        return context

    def handle_form_valid(self, form):
        reversed_url = reverse('so_fraud.lists_search',
                               kwargs={"project": self.kwargs['project']})

        query_params = {
            'channel': form.cleaned_data['channel'],
            'list_name': form.cleaned_data['list_name'],
            'value': form.cleaned_data.get('value'),
        }

        query_params = urllib.urlencode(query_params)

        return HttpResponseRedirect(reversed_url + '?' + query_params)

    def handle_get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)

        request_get_data = {k: urllib.unquote_plus(value) for k, value in self.request.GET.items()}

        list_name = request_get_data.get("list_name")

        if list_name:
            context['items'] = utils.get_lists(
                request_get_data["channel"],
                request_get_data["list_name"],
                request_get_data.get("value"),
            )

        context['now'] = utils.now_milliseconds()

        return self.render_to_response(context)


class TransactionsSearchView(StaffMemberFormView):
    template_name = 'so_fraud/transactions_template.html'
    form_class = TransactionsRequestForm

    def get_initial(self):
        initial = super(TransactionsSearchView, self).get_initial()
        return initial

    def handle_form_valid(self, form):
        reversed_url = reverse('so_fraud.transactions_search',
                               kwargs={"project": self.kwargs['project']})

        query_params = {
            'channel': json.dumps(form.cleaned_data.get('channel')),
            'query': form.cleaned_data.get('query'),
            'limit': form.data.get('limit'),
            'since': form.data.get('since'),
            'until': form.data.get('until'),
        }

        query_params = urllib.urlencode(query_params)

        return HttpResponseRedirect(reversed_url + '?' + query_params)

    def handle_get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)

        request_get_data = {k: urllib.unquote_plus(value) for k, value in self.request.GET.items()}

        query = request_get_data.get("query")

        if query is not None:
            if query:
                query = "(" + query + ") AND "

            query += utils.make_timestamp_query(
                datetime.strptime(request_get_data['since'], DATETIME_LOCAL_FORMAT),
                datetime.strptime(request_get_data['until'], DATETIME_LOCAL_FORMAT))

            channel = json.loads(request_get_data["channel"])
            items = utils.get_transactions_by_channels(
                channel if channel else self.get_form().channels,
                "MAIN",
                request_get_data["limit"],
                query)['transactions']

            context['items'] = items
            context['items_str'] = json.dumps(items)

        return self.render_to_response(context)


class TransactionView(StaffMemberView, TemplateView):
    template_name = 'so_fraud/transaction_template.html'

    def handle_get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)

        request_get_data = {k: urllib.unquote_plus(value) for k, value in self.request.GET.items()}

        tx_id = request_get_data['id']

        if tx_id.startswith('txn_'):
            tx_id = tx_id[4:]

        response = utils.get_transactions(request_get_data["channel"],
                                          "MAIN", 1,
                                          'id:"txn_' + tx_id + '"')

        if len(response['transactions']) > 0:
            context['transaction'] = response['transactions'][0]
            context['transaction_pretty_json'] = json.dumps(context['transaction'],
                                                            sort_keys=True,
                                                            indent=4)

        response = utils.get_transactions(request_get_data["channel"],
                                          "AGGRS", 1,
                                          'id:"txn_aggrs_' + tx_id + '"')

        if len(response['aggregates']) > 0:
            context['aggregates'] = response['aggregates'][0]
            context['aggregates_pretty_json'] = json.dumps(context['aggregates'],
                                                           sort_keys=True,
                                                           indent=4)
            if "user_context" in context['aggregates']:
                context['user_context_pretty_json'] = json.dumps(
                    context['aggregates']['user_context'],
                    sort_keys=True,
                    indent=4)

        return self.render_to_response(context)
