package ru.yandex.search.yc;

import java.io.IOException;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;

public enum MarketplaceFieldType {
    CATEGORY {
        @Override
        public String generateField(final String project, final String name) {
            return MarketplaceFieldType.generateExtField(project, "categories_id") +
                "tokenizer = lf\n" +
                "store = true\n" +
                "prefixed = true\n" +
                "attribute = true\n" +
                "analyze = true\n\n";
        }

        @Override
        public void writeIndex(
            final String project,
            final String name,
            final JsonWriter writer,
            final JsonObject value)
            throws IOException,JsonException
        {
            JsonList list = value.asList();
            StringBuilder catIds = new StringBuilder();
            StringBuilder catRanks = new StringBuilder();
            for (JsonObject categoryObj: list) {
                JsonMap category = categoryObj.asMap();
                String categoryId = category.getString("id");
                int rank = category.getInt("rank");
                catIds.append(categoryId);
                catIds.append("\n");
                catRanks.append(categoryId);
                catRanks.append('\t');
                catRanks.append(rank);
                catRanks.append('\n');
            }

            if (catIds.length() > 0) {
                catIds.setLength(catIds.length() - 1);
            }

            if (catRanks.length() > 0) {
                catRanks.setLength(catRanks.length() - 1);
            }

            writer.key(MarketplaceFieldType.generateExtFieldName(project, name + "_id"));
            writer.value(catIds);
            writer.key(MarketplaceFieldType.generateIntFieldName(project, name + "_map"));
            writer.value(catRanks);
        }
    },
    BLOB {
        @Override
        public String generateField(final String project, final String name) {
            return MarketplaceFieldType.generateExtField(project, name) +
                "store = true\n" +
                "index = false\n";
        }
    },
    INT {
        @Override
        public String generateField(final String project, final String name) {
            return MarketplaceFieldType.generateExtField(project, name) +
                "tokenizer = keyword\n" +
                "store = true\n" +
                "prefixed = true\n" +
                "attribute = true\n" +
                "analyze = true\n" +
                "type = long\n";
        }
    },
    TIMESTAMP {
        @Override
        public String generateField(final String project, final String name) {
            return MarketplaceFieldType.generateExtField(project, name) +
                "tokenizer = keyword\n" +
                "type = integer\n" +
                "store = true\n" +
                "prefixed = true\n" +
                "attribute = true\n" +
                "analyze = true\n";
        }
    },
    KEYWORD {
        @Override
        public String generateField(final String project, final String name) {
            return MarketplaceFieldType.generateExtField(project, name) +
                "tokenizer = keyword\n" +
                "store = true\n" +
                "prefixed = true\n" +
                "attribute = true\n" +
                "analyze = true\n";
        }
    },
    KEYWORD_ARRAY {
        @Override
        public String generateField(final String project, final String name) {
            return MarketplaceFieldType.generateExtField(project, name) +
                "tokenizer = lf\n" +
                "store = true\n" +
                "prefixed = true\n" +
                "attribute = true\n" +
                "analyze = true\n";
        }

        @Override
        public void writeIndex(
            final String project,
            final String name,
            final JsonWriter writer,
            final JsonObject value)
            throws IOException, JsonException
        {
            StringBuilder sb = new StringBuilder();
            for (JsonObject jo: value.asList()) {
                sb.append(jo.asString());
                sb.append("\n");
            }

            if (sb.length() > 0) {
                sb.setLength(sb.length() - 1);
            }

            writer.key(MarketplaceFieldType.generateExtFieldName(project, name));
            writer.value(sb.toString());
        }
    },
    FULL_TEXT {
        @Override
        public String generateField(final String project, final String name) {
            return MarketplaceFieldType.generateExtField(project, name) +
                "tokenizer = letter\n" +
                "filters = lowercase|replace:ё:е|lemmer\n" +
                "store = true\n" +
                "prefixed = true\n" +
                "attribute = true\n" +
                "analyze = true\n";
        }
    };

    public abstract String generateField(final String project, final String name);

    public void writeIndex(
        final String project,
        final String name,
        final JsonWriter writer,
        final JsonObject value)
        throws IOException,JsonException
    {
        writer.key(MarketplaceFieldType.generateExtFieldName(project, name));
        writer.value(value);
    }

    public static String generateExtField(final String project, final String name) {
        return "[field." + generateName(project, "yc_mkpl_ext", name) + "]\n";
    }

    public static String generateIntField(final String project, final String name) {
        return "[field." + generateName(project, "yc_mkpl_int", name) + "]\n";
    }

    public static String generateIntFieldName(final String project, final String name) {
        return generateName(project, "yc_mkpl_int", name);
    }

    public static String generateExtFieldName(final String project, final String name) {
        return generateName(project, "yc_mkpl_ext", name);
    }

    public static String generateName(final String project, final String baseName, final String name) {
        if (project.isEmpty()) {
            return baseName + "_" + name.replaceAll("\\.", "_");
        } else {
            return baseName + "_" + project + "__" + name.replaceAll("\\.", "_");
        }
    }

    public static String prefix(final String project) {
        if (project.isEmpty()) {
            return YcConstants.MARKETPLACE_PREFIX;
        } else {
            return YcConstants.MARKETPLACE_PREFIX + '_' + project;
        }
    }
}
