package ru.yandex.search.yc;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;

public abstract class AbstractIndexationTarget implements IndexationTarget {
    protected List<SearchBackendDoc> generateDocs(
        final YcDoc doc)
        throws JsonException
    {
        // main doc

        List<SearchBackendDoc> docs = new ArrayList<>();
        docs.add(new MainDoc(doc));


        JsonMap labels = null;
        for (Map.Entry<String, JsonObject> entry: doc.attributes().entrySet()) {
            if (YcIndexFields.LABELS.equalsIgnoreCase(entry.getKey())) {
                labels = entry.getValue().asMap();
                continue;
            }

            if (entry.getValue().type() == JsonObject.Type.NULL) {
                continue;
            }
            docs.add(new AttributeDoc(doc, entry));
        }

        if (labels != null) {
            for (Map.Entry<String, JsonObject> labelEntry: labels.entrySet()) {
                docs.add(new LabelDoc(doc, labelEntry));
            }
        }

        return docs;
    }

    protected StringBuilder deleteUri(final YcDoc doc, final StringPrefix prefix) {
        StringBuilder uri = new StringBuilder();
        uri.append("/update?delete_doc&prefix=");
        uri.append(prefix.toString());
        uri.append("&ts=");
        uri.append(doc.timestampMicros());
        uri.append("&resId=");
        uri.append(doc.resourceId());
        uri.append("&doc_service=");
        uri.append(doc.service());
        uri.append("&folder_id=");
        uri.append(doc.folderId());
        return uri;
    }

    protected StringBuilderWriter deleteDocJson(
        final YcDoc doc,
        final StringPrefix prefix)
        throws IOException
    {
        StringBuilderWriter updateSbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(updateSbw)) {
            StringBuilder text = new StringBuilder();
            text.append(YcFields.ID.prefixedField());
            text.append(':');
            text.append(YcFields.mainDocId(doc));
            text.append(" AND ");
            text.append(YcFields.TIMESTAMP.prefixedField());
            text.append(":[");
            text.append(doc.timestampMicros() + 1);
            text.append(" TO ");
            text.append(Long.MAX_VALUE);
            text.append(']');

            writer.startObject();
            writer.key("prefix");
            writer.value(prefix);
            writer.key("UpdateIfNotMatches");
            writer.value("true");
            writer.key("query");
            writer.value(text.toString());
            writer.key("AddIfNotExists");
            writer.value(true);
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            writer.key(YcFields.ID.storeField());
            writer.value(YcFields.mainDocId(doc));
            writer.key(YcFields.DOC_TYPE.storeField());
            writer.value(YcDocType.FULL_DOC.fieldValue());
            writer.key(YcFields.TIMESTAMP.storeField());
            writer.value(doc.timestampMicros());
            writer.key(YcFields.DELETED_TIME.storeField());
            writer.value(doc.timestampMicros());
            writer.endObject();
            writer.endArray();
            writer.endObject();
        }

        return updateSbw;
    }

    protected StringBuilder updateUri(final YcDoc doc, final StringPrefix prefix) {
        StringBuilder uri = new StringBuilder();
        uri.append("/update?update_doc&prefix=");
        uri.append(prefix.toString());
        uri.append("&ts=");
        uri.append(doc.timestampMicros());
        uri.append("&resId=");
        uri.append(doc.resourceId());
        uri.append("&doc_service=");
        uri.append(doc.service());
        uri.append("&folder_id=");
        uri.append(doc.folderId());

        return uri;
    }

    protected StringBuilderWriter updateJsonDoc(
        final YcDoc doc,
        final StringPrefix prefix)
        throws IOException, JsonException
    {
        List<SearchBackendDoc> docs = generateDocs(doc);
        StringBuilderWriter updateSbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(updateSbw)) {
            StringBuilder text = new StringBuilder();
            text.append(YcFields.ID.prefixedField());
            text.append(':');
            text.append(YcFields.mainDocId(doc));
            text.append(" AND ");
            text.append(YcFields.TIMESTAMP.prefixedField());
            text.append(":[");
            if (doc.reindex()) {
                text.append(doc.reindexTimestampMicros() + 1);
            } else {
                text.append(doc.timestampMicros() + 1);
            }

            text.append(" TO ");
            text.append(Long.MAX_VALUE);
            text.append(']');

            writer.startObject();
            writer.key("prefix");
            writer.value(prefix);
            writer.key("UpdateIfNotMatches");
            writer.value("true");
            writer.key("query");
            writer.value(text.toString());
            writer.key("AddIfNotExists");
            writer.value(true);
            writer.key("docs");
            writer.startArray();
            for (SearchBackendDoc backendDoc : docs) {
                writer.value(backendDoc);
            }

            writer.endArray();
            writer.endObject();
        }

        return updateSbw;
    }

    protected StringBuilderWriter cleanupEldersJsonDoc(
        final YcDoc doc,
        final StringPrefix prefix)
        throws IOException
    {
        StringBuilderWriter updateSbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(updateSbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(prefix);
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            writer.key(YcFields.TIMESTAMP.storeField());
            writer.value(doc.timestampMicros());
            writer.key(YcFields.DELETED_TIME.storeField());
            writer.value(doc.timestampMicros());
            writer.endObject();
            writer.endArray();
            writer.endObject();
        }

        return updateSbw;
    }

    protected QueryConstructor cleanupEldersUri(
        final YcDoc doc,
        final StringPrefix prefix)
        throws BadRequestException
    {
        StringBuilder text = new StringBuilder();
        text.append(YcFields.TIMESTAMP.prefixedField());
        text.append(":[0 TO ");
        text.append(doc.timestampMicros() - 1);
        text.append("] AND (");
        text.append(YcFields.ID.prefixedField());
        text.append(':');
        text.append(YcFields.mainDocId(doc));
        text.append(" OR ");
        text.append(YcFields.MAIN_DOC_ID.prefixedField());
        text.append(':');
        text.append(YcFields.mainDocId(doc));
        text.append(')');

        QueryConstructor qc = new QueryConstructor("/delete?cleanup_after_update");
        qc.append("text", text.toString());
        qc.append("prefix", prefix.toStringFast());
        qc.append("resId", doc.resourceId());
        qc.append("doc_service", doc.service());
        qc.append("folder_id", doc.folderId());

        return qc;
    }
}
