package ru.yandex.search.yc;

import java.io.IOException;
import java.util.List;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.json.parser.JsonException;
import ru.yandex.search.prefix.StringPrefix;

public class BackendIndexationTarget extends AbstractIndexationTarget {
    private final HttpHost backendHost;
    private final YcSearchIndexer indexer;

    public BackendIndexationTarget(final HttpHost backendHost, final YcSearchIndexer indexer) {
        this.backendHost = backendHost;
        this.indexer = indexer;
    }

    @Override
    public void accept(
        final YcSearchIndexRequestContext requestContext,
        final List<YcDoc> docs,
        final FutureCallback<Object> callback)
    {
        if (docs.size() <= 0) {
            callback.completed(null);
            return;
        }

        MultiFutureCallback<Object> mfcb = new MultiFutureCallback<>(callback);
        for (YcDoc doc: docs) {
            FutureCallback<Object> docCallback = mfcb.newCallback();
            try {
                accept(requestContext, doc, docCallback);
            } catch (BadRequestException | JsonException | IOException e) {
                docCallback.failed(e);
                mfcb.done();
                return;
            }

        }

        mfcb.done();
    }

    public void accept(
        final YcSearchIndexRequestContext requestContext,
        final YcDoc doc,
        final FutureCallback<Object> callback)
        throws BadRequestException, JsonException, IOException
    {
        ProxySession session = requestContext.session();
        StringPrefix prefix = new StringPrefix(YcFields.buildPrefix(doc.cloudId()));

        BasicAsyncRequestProducerGenerator post;
        if (doc.deleted()) {
            session.logger().fine("Deleting doc " + doc.resourceId() + " " + doc.service());
            indexer.deletes().accept(1);

            post = new BasicAsyncRequestProducerGenerator(
                deleteUri(doc, prefix).toString(),
                new StringEntity(
                    deleteDocJson(doc, prefix).toString(),
                    ContentType.APPLICATION_JSON));

            //post.addHeader(YandexHeaders.SERVICE, YcConstants.YC_QUEUE);
        } else {
            session.logger().fine("Updating doc " + doc.resourceId() + " " + doc.service());
            indexer.updates().accept(1);

            post = new BasicAsyncRequestProducerGenerator(
                updateUri(doc, prefix).toString(),
                new StringEntity(
                    updateJsonDoc(doc, prefix).toString(),
                    ContentType.APPLICATION_JSON));
        }

        indexer.acceptedByService().getOrCreate(doc.service()).accept(1L);
        requestContext.client().execute(
            indexer.server().producerHost(),
            post,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            session.listener().adjustContextGenerator(
                requestContext.client().httpClientContextGenerator()),
            new UpdateCallback(callback, doc, prefix, requestContext));
    }

    private class UpdateCallback extends FilterFutureCallback<Object> {
        private final YcDoc doc;
        private final StringPrefix prefix;
        private final YcSearchIndexRequestContext context;

        public UpdateCallback(
            final FutureCallback<? super Object> callback,
            final YcDoc doc,
            final StringPrefix prefix,
            final YcSearchIndexRequestContext context)
        {
            super(callback);
            this.doc = doc;
            this.prefix = prefix;
            this.context = context;
        }

        @Override
        public void completed(final Object result) {
            if (doc.reindex()) {
                callback.completed(null);
                return;
            }
            try {
                //System.out.println(cleanupEldersJsonDoc(doc, prefix).toString());
                BasicAsyncRequestProducerGenerator get = new BasicAsyncRequestProducerGenerator(
                    cleanupEldersUri(doc, prefix).toString());

                context.client().execute(
                    indexer.server().producerHost(),
                    get,
                    BasicAsyncResponseConsumerFactory.ANY_GOOD,
                    context.session().listener().adjustContextGenerator(
                        context.client().httpClientContextGenerator()),
                    callback);
            } catch (BadRequestException e) {
                failed(e);
            }
        }
    }

}
