package ru.yandex.search.yc;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.search.yc.config.ImmutableFieldGroupingElementStaterConfig;
import ru.yandex.search.yc.config.ImmutableFieldGroupingElementStatersConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class FieldGroupingStater extends TimerTask implements Stater {
    private final AsyncClient client;
    private final HttpHost host;
    private final PrefixedLogger logger;
    private final Timer timer;
    private final AtomicReference<Map<String, Long>> stat
        = new AtomicReference<>(Collections.emptyMap());
    private final ImmutableFieldGroupingElementStatersConfig config;
    private final String fieldName;

    public FieldGroupingStater(
        final UniversalSearchProxy<?> proxy,
        final ImmutableFieldGroupingElementStatersConfig config,
        final HttpHost backendHost,
        final String fieldName)
    {
        this.logger = proxy.logger().addPrefix("FieldGroupingStater");
        this.client = proxy.client("FieldGroupingStatClient " + fieldName, config);
        this.host = backendHost;

        this.timer = new Timer("FieldGroupingStaterTimer", true);
        this.config = config;
        this.fieldName = fieldName;
    }

    public void start() {
        this.timer.scheduleAtFixedRate(this, 1000, config.interval());
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Map<String, Long> data = stat.get();
        for (Map.Entry<String, Long> item: data.entrySet()) {
            statsConsumer.stat(item.getKey(), item.getValue());
        }
    }

    private long fieldGroupingElementStat(final ImmutableFieldGroupingElementStaterConfig config, final String fieldValue) throws InterruptedException, HttpException {
        try {
            QueryConstructor qc = new QueryConstructor("/search?field_grouping_stater");
            String text = this.fieldName + ":" + (fieldValue.equals("") ? "default" : fieldValue);
            if (!config.query().isBlank()) {
                text += " AND (" + config.query() + ")";
            }
            qc.append("text", text);
            qc.append("length", "1");
            qc.append("IO_PRIO", Integer.MAX_VALUE - 1);
            qc.append("get", "id");

            Future<JsonObject> future = client.execute(
                host,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                client.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            JsonMap searchResult = future.get(this.config.interval(), TimeUnit.MILLISECONDS).asMap();
            return searchResult.getLong("hitsCount");
        } catch (TimeoutException | ExecutionException | JsonException e) {
            throw new HttpException("Failed to fetch fieldName:" + fieldValue, e);
        }
    }

    private Collection<String> fieldValues(String fieldName) throws InterruptedException, HttpException {
        try {
            Future<JsonObject> future = client.execute(
                host,
                new BasicAsyncRequestProducerGenerator(
                    "/printkeys?field=" + fieldName + "&hr&source=field_grouping_stater"),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                client.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            JsonMap printkeysResult = future.get(config.interval(), TimeUnit.MILLISECONDS).asMap();
            return new LinkedHashSet<>(printkeysResult.keySet());
        } catch (TimeoutException | ExecutionException | JsonException e) {
            throw new HttpException("Failed to fetch fieldName values", e);
        }
    }

    @Override
    public void run() {
        logger.info("Updating " + fieldName + " stat");
        try {
            Collection<String> fieldValues = fieldValues(this.fieldName);
            logger.info(fieldName + " values loaded: " + String.join(",", fieldValues));
            Map<String, Long> result = new LinkedHashMap<>();
            for (String fieldValue: fieldValues) {
                for (Map.Entry<String, ImmutableFieldGroupingElementStaterConfig> entry: config.fieldGroupingElementStats().entrySet()) {
                    try {
                        long count = fieldGroupingElementStat(entry.getValue(), fieldValue);
                        result.put(entry.getValue().signalName().replaceFirst("\\{field_value\\}", fieldValue), count);
                        logger.info(entry.getKey() + " value " + count);
                    } catch (HttpException he) {
                        logger.log(
                            Level.WARNING,
                            "Computation failed for " + fieldName  + ":" + fieldValue + " name " + entry.getKey() + " query " + entry.getValue().query(),
                            he);
                    }
                }

            }

            stat.set(result);
        } catch (Exception e) {
            logger.log(Level.WARNING, "Failed to calculate " + fieldName + " stat", e);
        }
        logger.info("Updating " + fieldName + " finished");
    }
}
