package ru.yandex.search.yc;

import java.io.IOException;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.dispatcher.producer.SearchMap;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;

public class ProducerIndexationTarget extends AbstractIndexationTarget {
    private static final StringBody EMPTY_BODY =
        new StringBody("", ContentType.TEXT_PLAIN);

    private final YcSearchIndexer indexer;

    public ProducerIndexationTarget(final YcSearchIndexer indexer) {
        this.indexer = indexer;
    }

    @Override
    public void accept(
        final YcSearchIndexRequestContext requestContext,
        final List<YcDoc> docs,
        final FutureCallback<Object> callback)
    {
        if (docs.size() <= 0) {
            callback.completed(null);
            return;
        }

        MultiFutureCallback<Object> mfcb = new MultiFutureCallback<>(callback);
        for (YcDoc doc: docs) {
            FutureCallback<Object> docCallback = mfcb.newCallback();
            try {
                accept(requestContext, doc, docCallback);
            } catch (BadRequestException | JsonException | IOException e) {
                docCallback.failed(e);
                mfcb.done();
                return;
            }

        }

        mfcb.done();
    }

    public void accept(
        final YcSearchIndexRequestContext requestContext,
        final YcDoc doc,
        final FutureCallback<Object> callback)
        throws BadRequestException, JsonException, IOException
    {
        ProxySession session = requestContext.session();
        StringPrefix prefix = new StringPrefix(YcFields.buildPrefix(doc.cloudId()));

        QueryConstructor qc = new QueryConstructor("/index_search_doc?");
        qc.append("ts", doc.timestampMicros());
        qc.append("cloud", doc.cloudId());
        qc.append("resId", doc.resourceId());
        qc.append("ycService", doc.service());

        MultipartEntityBuilder builder =
            MultipartEntityBuilder.create();
        builder.setMimeSubtype("mixed");

        if (doc.deleted()) {
            session.logger().fine("Deleting doc " + doc.resourceId() + " " + doc.service());
            qc.append("deleted", doc.deleteTsMicros());
            delete(builder, session.logger(), doc, prefix);
            indexer.deletes().accept(1);
        } else {
            session.logger().fine("Updating doc " + doc.resourceId() + " " + doc.service());
            updateRequest(builder, session.logger(), doc, prefix);
            indexer.updates().accept(1);
        }

        indexer.acceptedByService().getOrCreate(doc.service()).accept(1L);
        if (!doc.reindex()) {
            cleanupElders(builder, doc, prefix);
        }

        BasicAsyncRequestProducerGenerator post = new BasicAsyncRequestProducerGenerator(
            qc.toString(),
            builder.build());

        post.addHeader(YandexHeaders.SERVICE, YcConstants.YC_QUEUE);

        requestContext.client().execute(
            indexer.server().producerHost(),
            post,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            session.listener().adjustContextGenerator(
                requestContext.client().httpClientContextGenerator()),
            callback);
    }

    private void delete(
        final MultipartEntityBuilder builder,
        final PrefixedLogger logger,
        final YcDoc doc,
        final StringPrefix prefix)
        throws BadRequestException, IOException
    {
        StringBuilder uri = deleteUri(doc, prefix);
        String shardId = String.valueOf(prefix.hash() % SearchMap.SHARDS_COUNT);
        StringBuilderWriter updateSbw = deleteDocJson(doc, prefix);

        builder.addPart(
            FormBodyPartBuilder.create()
                .addField(YandexHeaders.URI, uri.toString())
                .addField(YandexHeaders.ZOO_SHARD_ID, shardId)
                .setBody(new StringBody(updateSbw.toString(), ContentType.APPLICATION_JSON))
                .setName("envelope.json")
                .build());

        logger.info("IndexMessage " + updateSbw.toString());

        indexer.outgoingLogger().info(updateSbw.toString());
    }

    private void updateRequest(
        final MultipartEntityBuilder builder,
        final PrefixedLogger logger,
        final YcDoc doc,
        final StringPrefix prefix)
        throws JsonException, IOException
    {
        String shardId = String.valueOf(prefix.hash() % SearchMap.SHARDS_COUNT);
        StringBuilder uri = updateUri(doc, prefix);
        StringBuilderWriter updateSbw = updateJsonDoc(doc, prefix);

        builder.addPart(
            FormBodyPartBuilder.create()
                .addField(YandexHeaders.URI, uri.toString())
                .addField(YandexHeaders.ZOO_SHARD_ID, shardId)
                .setBody(new StringBody(updateSbw.toString(), ContentType.APPLICATION_JSON))
                .setName("envelope.json")
                .build());

        indexer.outgoingLogger().info(updateSbw.toString());
    }

    private void cleanupElders(
        final MultipartEntityBuilder builder,
        final YcDoc doc,
        final StringPrefix prefix)
        throws BadRequestException, IOException
    {
        StringBuilderWriter updateSbw = cleanupEldersJsonDoc(doc, prefix);

        QueryConstructor qc = cleanupEldersUri(doc, prefix);

        builder.addPart(
            FormBodyPartBuilder.create()
                .addField(YandexHeaders.URI, qc.toString())
                .addField(YandexHeaders.ZOO_HTTP_METHOD, "GET")
                .setBody(EMPTY_BODY)
                .addField(
                    YandexHeaders.ZOO_SHARD_ID,
                    String.valueOf(prefix.hash() % SearchMap.SHARDS_COUNT))
                .setName("envelope.json").build());

        //outgoingLogger.info("{\"action\": \"delete\", \"ts\":\"" + doc.deletedTsStr() + "\"}");
    }
}
