package ru.yandex.search.yc;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.search.yc.config.ImmutableFieldGroupingElementStatersConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

/**
 * Two stale thresholds
 *  1) >= 2 days - show on graphic + allow services to receive by calling method on proxy
 *  2) >= 14 days mark stale in index and stop returning in search methods
 */
public class ReindexWarden extends TimerTask implements Stater {
    private static final long MAGIC_VALUE = 100500L;
    private static final long STALE_THRESHOLD = TimeUnit.DAYS.toMicros(2);
    private static final long SUSPICIOUS_THRESHOLD =
        TimeUnit.DAYS.toMicros(1) + TimeUnit.HOURS.toMicros(1);
    private static final long LAST_REINDEX_TOO_OLD_THRESHOLD = TimeUnit.DAYS.toMicros(8);
    private static final int MARK_STALE_PREFETCH = 10000;

    private final AsyncClient client;
    private final HttpHost searchHost;
    private final HttpHost indexHost;
    private final PrefixedLogger logger;
    private final Timer timer;
    private final ImmutableFieldGroupingElementStatersConfig config;

    private final AtomicReference<Stat> statReference
        = new AtomicReference<>(
            new Stat(Collections.emptyMap(), Collections.emptyMap(), Collections.emptyMap(), Collections.emptyMap()));

    public ReindexWarden(
        final UniversalSearchProxy<?> proxy,
        final ImmutableFieldGroupingElementStatersConfig config,
        final HttpHost indexHost,
        final HttpHost searchHost)
    {
        this.logger = proxy.logger().addPrefix("ReindexWarden");
        this.client = proxy.searchClient();
        this.searchHost = searchHost;
        this.indexHost = indexHost;

        this.config = config;
        this.timer = new Timer("ReindexWardenTimer", true);
    }

    public void start() {
        //run();
        this.timer.scheduleAtFixedRate(this, 100, config.interval());
    }

    public Map<String, Object> status(
        final boolean verbose)
    {
        if (!verbose) {
            return Collections.emptyMap();
        }

        Stat stat = statReference.get();

        return stat.status();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Stat stat = statReference.get();
        long total = 0;
        for (Map.Entry<String, Long> item: stat.staleDocsCount().entrySet()) {
            if (item.getValue() != MAGIC_VALUE && item.getValue() > 0) {
                total += item.getValue();
            }
            statsConsumer.stat("stale_docs_in_index_" + item.getKey() + "_axxx", item.getValue());
        }
        long suspTotal = 0;
        for (Map.Entry<String, Long> item: stat.suspiciousDocsCount().entrySet()) {
            if (item.getValue() != MAGIC_VALUE && item.getValue() > 0) {
                suspTotal += item.getValue();
            }

            statsConsumer.stat("suspicious_stale_docs_in_index_" + item.getKey() + "_axxx", item.getValue());
        }

        statsConsumer.stat("stale_docs_in_index_total_axxx", total);
        statsConsumer.stat("suspicious_docs_in_index_total_axxx", suspTotal);
        statsConsumer.stat("services_count_reindex_actual_axxx", stat.servicesReindexActual());
        statsConsumer.stat("services_count_reindex_stale_axxx", stat.servicesNoReindex());
        statsConsumer.stat("stale_docs_marked_axxx", stat.markedStale());
    }

    private long extraDocumentsByService(final String service, final long ts) throws InterruptedException, HttpException {
        try {
            QueryConstructor qc = new QueryConstructor("/search?service_stater");
            StringBuilder textSb = new StringBuilder();
            textSb.append("yc_service:");
            textSb.append(service);
            textSb.append(" AND yc_reindex_timestamp:[0 TO ");
            textSb.append(ts);
            textSb.append("]");
            textSb.append(" AND yc_doc_type:main AND NOT yc_deleted:1 AND NOT yc_stale:1");
            qc.append("text", textSb.toString());
            qc.append("length", "100");
            qc.append("IO_PRIO", Integer.MAX_VALUE - 1);
            qc.append("get", "*");

            Future<JsonObject> future = client.execute(
                searchHost,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                client.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            JsonMap searchResult = future.get(config.interval(), TimeUnit.MILLISECONDS).asMap();
            return searchResult.getLong("hitsCount");
        } catch (TimeoutException | ExecutionException | JsonException e) {
            throw new HttpException("Failed to fetch services", e);
        }
    }

    private long suspiciousDocumentsByService(final String service, final long firstTs, final long lastTs) throws InterruptedException, HttpException {
        try {
            QueryConstructor qc = new QueryConstructor("/search?service_stater");
            StringBuilder textSb = new StringBuilder();
            textSb.append("yc_service:");
            textSb.append(service);
            textSb.append(" AND yc_reindex_timestamp:[");
            textSb.append(firstTs);
            textSb.append(" TO ");
            textSb.append(lastTs);
            textSb.append("]");
            textSb.append(" AND yc_doc_type:main AND NOT yc_deleted:1 AND NOT yc_stale:1");
            qc.append("text", textSb.toString());
            qc.append("length", "100");
            qc.append("IO_PRIO", Integer.MAX_VALUE - 1);
            qc.append("get", "*");

            Future<JsonObject> future = client.execute(
                searchHost,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                client.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            JsonMap searchResult = future.get(config.interval(), TimeUnit.MILLISECONDS).asMap();
            return searchResult.getLong("hitsCount");
        } catch (TimeoutException | ExecutionException | JsonException e) {
            throw new HttpException("Failed to fetch services", e);
        }
    }

    private Collection<String> services() throws InterruptedException, HttpException {
        try {
            Future<JsonObject> future = client.execute(
                searchHost,
                new BasicAsyncRequestProducerGenerator(
                    "/printkeys?field=yc_service&hr&source=service_stater"),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                client.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            JsonMap printkeysResult = future.get(config.interval(), TimeUnit.MILLISECONDS).asMap();
            return new LinkedHashSet<>(printkeysResult.keySet());
        } catch (TimeoutException | ExecutionException | JsonException e) {
            throw new HttpException("Failed to fetch services", e);
        }
    }

    protected SingleServiceStat processService(final String service, final Stat stat, final long nowMicros) {
        try {
            long reindexTimestamp = serviceReindexTimestamp(service);
            long mostEarlyValidTs = reindexTimestamp - STALE_THRESHOLD;
            long extra = MAGIC_VALUE;
            long suspicious = MAGIC_VALUE;
            if (nowMicros - reindexTimestamp < 0) {
                logger.warning("Strange reindex ts for service " + service + " " + mostEarlyValidTs);
                extra = MAGIC_VALUE;
                suspicious = MAGIC_VALUE;
            } else {
                // now we fetching all stale docs for services with active reindex
                if (nowMicros - reindexTimestamp < LAST_REINDEX_TOO_OLD_THRESHOLD) {
                    extra = extraDocumentsByService(service, mostEarlyValidTs);
                    suspicious = suspiciousDocumentsByService(
                        service,
                        reindexTimestamp - STALE_THRESHOLD,
                        reindexTimestamp - SUSPICIOUS_THRESHOLD);
                } else {
                    extra = -1;
                    suspicious = -1;
                }
            }

            return new SingleServiceStat(reindexTimestamp, extra, suspicious);
        } catch (Exception e) {
            logger.log(Level.WARNING, "Stat failed for service " + service);
            return new SingleServiceStat(-1, -1, -1);
        }
    }

    private static final class SingleServiceStat {
        private final long reindexTimestamp;
        private final long stale;
        private final long suspicious;

        public SingleServiceStat(long reindexTimestamp, long stale, long suspicious) {
            this.reindexTimestamp = reindexTimestamp;
            this.stale = stale;
            this.suspicious = suspicious;
        }
    }

    protected long serviceReindexTimestamp(final String service) throws Exception {
        QueryConstructor qc = new QueryConstructor("/search?service_stater");
        String text = "yc_service:" + service;
        qc.append("text", text);
        qc.append("length", "100");
        qc.append("merge_func", "none");
        qc.append("sort", "yc_reindex_timestamp");
        qc.append("IO_PRIO", Integer.MAX_VALUE - 1);
        qc.append("get", "yc_service,yc_reindex_timestamp");

        Future<JsonObject> future = client.execute(
            searchHost,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            client.httpClientContextGenerator(),
            EmptyFutureCallback.INSTANCE);
        JsonMap searchResult = future.get(config.interval(), TimeUnit.MILLISECONDS).asMap();
        JsonList hitsArray = searchResult.getList("hitsArray");

        for (JsonObject itemObj: hitsArray) {
            JsonMap item = itemObj.asMap();
            long reindexTimestamp = item.getLong("yc_reindex_timestamp", -1L);

            if (reindexTimestamp > 0) {
                return reindexTimestamp;
            }
        }

        return -1L;
    }

    public Map<String, Long> servicesReindexTimestamps() throws InterruptedException, HttpException {
        try {
            Collection<String> services = services();

            Map<String, Long> reindexTimestamps = new LinkedHashMap<>();
            for (String service: services) {
                QueryConstructor qc = new QueryConstructor("/search?service_stater");
                String text = "yc_service:" + service;
                qc.append("text", text);
                qc.append("length", "100");
                qc.append("merge_func", "none");
                qc.append("sort", "yc_reindex_timestamp");
                qc.append("IO_PRIO", Integer.MAX_VALUE - 1);
                qc.append("get", "yc_service,yc_reindex_timestamp");

                Future<JsonObject> future = client.execute(
                    searchHost,
                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    client.httpClientContextGenerator(),
                    EmptyFutureCallback.INSTANCE);
                JsonMap searchResult = future.get(config.interval(), TimeUnit.MILLISECONDS).asMap();
                JsonList hitsArray = searchResult.getList("hitsArray");

                for (JsonObject itemObj: hitsArray) {
                    JsonMap item = itemObj.asMap();
                    long reindexTimestamp = item.getLong("yc_reindex_timestamp", -1L);

                    if (reindexTimestamp > 0) {
                        reindexTimestamps.put(service, reindexTimestamp);
                    }
                }
            }

            return reindexTimestamps;
        } catch (TimeoutException | ExecutionException | JsonException e) {
            throw new HttpException("Failed to fetch services", e);
        }
    }

    private boolean sendMarks(
        final String service,
        final String prefix,
        final JsonList list,
        final int first,
        final int last,
        final long ts,
        final long total)
    {
        try {
            StringBuilderWriter sbw = new StringBuilderWriter();
            JsonWriter writer = JsonType.NORMAL.create(sbw);

            QueryConstructor qc = new QueryConstructor("/update?stale_marker");
            qc.append("prefix", prefix);
            qc.append("yc_service", service);
            qc.append("cnt", last - first + 1);

            writer.startObject();
            writer.key("prefix");
            writer.value(prefix);
            writer.key("docs");
            writer.startArray();
            for (int i = first; i <= last; i++) {
                JsonObject jo = list.get(i);
                String id = jo.asMap().getString("id");
                writer.startObject();
                writer.key("id");
                writer.value(id);
                writer.key("yc_marked_stale_ts");
                writer.value(ts);
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
            writer.close();
            String data = sbw.toString();
            logger.info("Updating stale " + data);

            client.execute(
                indexHost,
                new BasicAsyncRequestProducerGenerator(qc.toString(), data),
                BasicAsyncResponseConsumerFactory.ANY_GOOD,
                client.httpClientContextGenerator(),
                new MarkStaleCallback(service, total - last + 1)).get(config.interval(), TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            logger.log(Level.WARNING, "Failed to mark stale " + service, e);
            return false;
        }

        return true;
    }

    private long markStaleDocs(
        final String service,
        final long reindexTs,
        final long toleranceTime)
        throws InterruptedException, HttpException
    {
        logger.info(
            "Marking stale for service "
                + service + " with reindex ts "
                + reindexTs + " and tolerance time " + toleranceTime);
        long ts = reindexTs - toleranceTime;
        try {
            QueryConstructor qc = new QueryConstructor("/search?service_stater");
            StringBuilder textSb = new StringBuilder();
            textSb.append("(yc_service:");
            textSb.append(service);
            textSb.append(" AND NOT yc_deleted:1 AND NOT yc_stale:1");
            textSb.append(" AND yc_reindex_timestamp:[0 TO ");
            textSb.append(ts);
            textSb.append("])");
            textSb.append(" OR (yc_service:");
            textSb.append(service);
            textSb.append(" AND NOT yc_deleted:1 AND NOT yc_stale:1");
            textSb.append(" AND NOT yc_reindex_timestamp:* AND NOT yc_timestamp:[");
            textSb.append(reindexTs - 1);
            textSb.append(" TO " + Long.MAX_VALUE + "]");
            textSb.append(")");
            qc.append("text", textSb.toString());
            qc.append("length", MARK_STALE_PREFETCH);
            qc.append("IO_PRIO", Integer.MAX_VALUE - 1);
            qc.append("get", "id,__prefix");

            Future<JsonObject> future = client.execute(
                searchHost,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                client.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            JsonMap searchResult = future.get(config.interval(), TimeUnit.MILLISECONDS).asMap();
            long total = searchResult.getLong("hitsCount");
            JsonList list = searchResult.getList("hitsArray");
            if (list.size() == 0) {
                logger.info("Nothing to mark stale for " + service);
                return 0L;
            } else {
                logger.info("Loaded for marking stale " + total + " for service " + service);
            }

            StringBuilderWriter sbw = new StringBuilderWriter();
            JsonWriter writer = JsonType.NORMAL.create(sbw);
            long nowMicros = 1000 * System.currentTimeMillis();
            String prefix = list.get(0).asMap().getString("__prefix");
            writer.startObject();
            writer.key("prefix");
            writer.value(prefix);
            writer.key("docs");
            writer.startArray();
            int first = 0;
            int i;
            for (i = 0; i < list.size(); i++) {
                JsonObject jo = list.get(i);
                String itemPrefix = jo.asMap().getString("__prefix");
                if (!prefix.equals(itemPrefix)) {
                    if (!sendMarks(service, prefix, list, first, i - 1, nowMicros, total)) {
                        return first + 1L;
                    }
                    first = i;
                    prefix = itemPrefix;
                }
            }

            sendMarks(service, prefix, list, first, list.size() - 1, nowMicros, total);

            return list.size();
        } catch (TimeoutException | IOException | ExecutionException | JsonException e) {
            throw new HttpException("Failed to fetch services", e);
        }
    }

    private  class MarkStaleCallback implements FutureCallback<Object> {
        private final String service;
        private final long left;

        public MarkStaleCallback(final String service, final long left) {
            this.service = service;
            this.left = left;
        }

        @Override
        public void completed(final Object o) {
            logger.info("Marking stale for service " + service + " completed, left docs: " + left);
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.WARNING, "Marking stale for service " + service + " failed, left docs: " + left, e);
        }

        @Override
        public void cancelled() {
            logger.warning("Marking stale for service " + service + " cancelled, left docs: " + left);
        }
    }

    @Override
    public void run() {
        logger.info("Updating services stat");
        try {
            Map<String, Long> servicesReindexTs = servicesReindexTimestamps();
            logger.info("Services with reindex ts loaded: " + servicesReindexTs);
            long nowMicros = 1000 * System.currentTimeMillis();

            Map<String, Long> extras = new LinkedHashMap<>();
            Map<String, Long> suspicious = new LinkedHashMap<>();

            for (Map.Entry<String, Long> entry: servicesReindexTs.entrySet()) {
                long mostEarlyValidTs = entry.getValue() - STALE_THRESHOLD;
                if (nowMicros - entry.getValue() < 0) {
                    logger.warning("Strange reindex ts for service " + entry.getKey() + " " + entry.getValue());
                    extras.put(entry.getKey(), MAGIC_VALUE);
                    suspicious.put(entry.getKey(), MAGIC_VALUE);
                } else {
                    // now we fetching all stale docs for services with active reindex
                    if (nowMicros - entry.getValue() < LAST_REINDEX_TOO_OLD_THRESHOLD) {
                        long count = extraDocumentsByService(entry.getKey(), mostEarlyValidTs);
                        extras.put(entry.getKey(), count);
                        long suspCount = suspiciousDocumentsByService(
                            entry.getKey(),
                            entry.getValue() - STALE_THRESHOLD,
                            entry.getValue() - SUSPICIOUS_THRESHOLD);
                        suspicious.put(entry.getKey(), suspCount);
                    } else {
                        extras.put(entry.getKey(), -1L);
                        suspicious.put(entry.getKey(), -1L);
                    }
                }
            }

            Stat stat = new Stat(extras, suspicious, servicesReindexTs, Collections.emptyMap());
            statReference.set(stat);

            Map<String, Long> markedStat = new LinkedHashMap<>();
            logger.info("Stat: " + stat);
            try {
                for (Map.Entry<String, Long> entry: extras.entrySet()) {
                    if (entry.getValue() > 0) {
                        long cnt = markStaleDocs(entry.getKey(), servicesReindexTs.get(entry.getKey()), STALE_THRESHOLD);
                        markedStat.put(entry.getKey(), cnt);
                    }
                }
            } finally {
                stat = new Stat(extras, suspicious, servicesReindexTs, markedStat);
                statReference.set(stat);
            }

        } catch (Exception e) {
            logger.log(Level.WARNING, "Failed to calculate services stat", e);
        }
    }

    private static final class Stat {
        private final Map<String, Long> staleDocsCount;
        private final Map<String, Long> suspiciousDocsCount;
        private final Map<String, Long> lastReindexByService;
        private final Map<String, Long> markedStaleByService;
        private final long servicesReindexActual;
        private final long servicesNoReindex;
        private final long markedStale;


        public Stat(
            final Map<String, Long> staleDocsCount,
            final Map<String, Long> suspiciousDocsCount,
            final Map<String, Long> lastReindexByService,
            final Map<String, Long> markedStaleByService)
        {
            this.staleDocsCount = staleDocsCount;
            this.suspiciousDocsCount = suspiciousDocsCount;
            this.lastReindexByService = lastReindexByService;
            this.markedStaleByService = markedStaleByService;
            long nowMicros = 1000 * System.currentTimeMillis();
            this.servicesReindexActual =
                lastReindexByService.values()
                    .stream().filter(ts -> (nowMicros - ts) < LAST_REINDEX_TOO_OLD_THRESHOLD).count();
            this.markedStale = markedStaleByService.values().stream().mapToLong(Long::longValue).sum();
            this.servicesNoReindex = lastReindexByService.values()
                .stream().filter(ts -> (nowMicros - ts) > LAST_REINDEX_TOO_OLD_THRESHOLD).count();
        }

        public long markedStale() {
            return markedStale;
        }

        public Map<String, Long> markedStaleByService() {
            return markedStaleByService;
        }

        public Map<String, Long> staleDocsCount() {
            return staleDocsCount;
        }

        public Map<String, Long> suspiciousDocsCount() {
            return suspiciousDocsCount;
        }

        public Map<String, Long> lastReindexByService() {
            return lastReindexByService;
        }

        public long servicesReindexActual() {
            return servicesReindexActual;
        }

        public long servicesNoReindex() {
            return servicesNoReindex;
        }

        public Map<String, Object> status() {
            long nowMicros = 1000 * System.currentTimeMillis();
            Map<String, Object> result = new LinkedHashMap<>();
            result.put("stale_docs_count", staleDocsCount);
            result.put("suspicious_docs_count", suspiciousDocsCount);
            result.put("service_last_reindex_timestamp", lastReindexByService);
            result.put("services_no_reindex", lastReindexByService.values()
                .stream().filter(ts -> (nowMicros - ts) > LAST_REINDEX_TOO_OLD_THRESHOLD).collect(Collectors.toList()));
            return result;
        }

        @Override
        public String toString() {
            return "Stat{" +
                "staleDocsCount=" + staleDocsCount +
                " suspicious=" + suspiciousDocsCount +
                ", lastReindexByService=" + lastReindexByService +
                ", servicesReindexActual=" + servicesReindexActual +
                ", servicesNoReindex=" + servicesNoReindex +
                '}';
        }
    }

}
