package ru.yandex.search.yc;

import java.io.IOException;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Map;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.yc.search.YcSearchFields;

public class YcIndexDoc implements JsonValue, YcDoc {
    private final String service;
    private final String timestampStr;
    private final String deletedTsStr;
    private final ZonedDateTime timestamp;
    private final long timestampMicros;
    private final long deleteTsMicros;
    private final String cloudId;
    private final String name;
    private final String folderId;
    private final String permission;
    private final String resourceId;
    private final String resourceType;
    private final Map<String, JsonObject> attributes;
    private final long transferTs;
    private final boolean deleted;

    public YcIndexDoc(final JsonMap doc, final long transferTs) throws JsonException {
        deletedTsStr = doc.getString(YcIndexFields.DELETED, "");
        service = doc.get(YcIndexFields.SERVICE, NonEmptyValidator.TRIMMED);
        timestampStr = doc.getString(YcIndexFields.TIMESTAMP);
        timestamp = ZonedDateTime.parse(timestampStr);
        name = doc.getString(YcIndexFields.NAME, null);
        cloudId = doc.get(YcIndexFields.CLOUD_ID, NonEmptyValidator.TRIMMED);
        folderId = doc.getString(YcIndexFields.FOLDER_ID);
        permission = doc.getString(YcIndexFields.PERMISSION);
        resourceId = doc.get(YcIndexFields.RESOURCE_ID, NonEmptyValidator.TRIMMED);
        resourceType = doc.get(YcIndexFields.RESOURCE_TYPE, NonEmptyValidator.TRIMMED);
        Map<String, JsonObject> attributes = doc.getMapOrNull(YcIndexFields.ATTRIBUTES);
        if (attributes == null) {
            attributes = Collections.emptyMap();
        }

        this.attributes = attributes;
        timestampMicros = ChronoUnit.MICROS.between(Instant.EPOCH, timestamp.toInstant());
        if (!deletedTsStr.isEmpty()) {
            ZonedDateTime deleteTs = ZonedDateTime.parse(deletedTsStr);
            deleted = true;
            deleteTsMicros = ChronoUnit.MICROS.between(Instant.EPOCH, deleteTs.toInstant());
        } else {
            deleteTsMicros = -1L;
            deleted = false;
        }

        this.transferTs = transferTs;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.key(YcFields.SERVICE.storeField());
        writer.value(service);
        writer.key(YcFields.CLOUD_ID.storeField());
        writer.value(cloudId);
        writer.key(YcFields.FOLDER_ID.storeField());
        writer.value(folderId);
        writer.key(YcFields.PERMISSION.storeField());
        writer.value(permission);
        writer.key(YcFields.RESOURCE_ID.storeField());
        writer.value(resourceId);
        writer.key(YcFields.RESOURCE_TYPE.storeField());
        writer.value(resourceType);
        writer.key(YcFields.TIMESTAMP_STR.storeField());
        writer.value(timestampStr);
        writer.key(YcFields.TIMESTAMP.storeField());
        writer.value(timestampMicros);
        writer.key(YcFields.TRANSFER_TS.storeField());
        writer.value(transferTs);
        writer.key(YcFields.DELETED_TIME.storeField());
        writer.value(JsonNull.INSTANCE);
        writer.key(YcSearchFields.YC_NAME.stored());
        writer.value(name);
        writer.key(YcSearchFields.YC_SCHEMA_VERSION.stored());
        writer.value(1);
        // we got update, so it is not  stale anymore
        writer.key(YcSearchFields.YC_MARKED_STALE_TS.stored());
        writer.value(JsonNull.INSTANCE);
        // second value
    }

    public String deletedTsStr() {
        return deletedTsStr;
    }

    @Override
    public long timestampMicros() {
        return timestampMicros;
    }

    @Override
    public long deleteTsMicros() {
        return deleteTsMicros;
    }

    @Override
    public boolean deleted() {
        return deleted;
    }

    @Override
    public String service() {
        return service;
    }

    @Override
    public String timestampStr() {
        return timestampStr;
    }

    @Override
    public ZonedDateTime timestamp() {
        return timestamp;
    }

    @Override
    public String cloudId() {
        return cloudId;
    }

    @Override
    public String folderId() {
        return folderId;
    }

    @Override
    public String permission() {
        return permission;
    }

    @Override
    public String resourceId() {
        return resourceId;
    }

    @Override
    public String resourceType() {
        return resourceType;
    }

    @Override
    public Map<String, JsonObject> attributes() {
        return attributes;
    }

    public long timestampMillis() {
        return timestampMicros;
    }

    public long transferTs() {
        return transferTs;
    }

    @Override
    public boolean reindex() {
        return false;
    }

    @Override
    public ZonedDateTime reindexTimestamp() {
        return null;
    }
}
