package ru.yandex.search.yc;

import java.io.IOException;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.yc.search.YcSearchFields;

public class YcIndexDocV2 implements JsonValue, YcDoc {
    private final String resourceType;
    private final long timestampMicros;
    private final ZonedDateTime timestamp;
    private final String timestampStr;
    private final String resourceId;
    private final String name;
    private final String service;
    private final String permission;
    private final String cloudId;
    private final String folderId;

    private final String deletedTsStr;
    private final long deleteTsMicros;
    private final boolean deleted;

    private final ZonedDateTime reindexTimestamp;
    private final long reindexTimestampMicros;
    private final Map<String, JsonObject> attributes;
    private final List<Map.Entry<String, String>> resourcePath;
    private final JsonList resourcePathJson;

    private final long transferTs;

    public YcIndexDocV2(final JsonMap doc, final long transferTs) throws JsonException {
        resourceType = doc.get(YcIndexFields.RESOURCE_TYPE, NonEmptyValidator.TRIMMED);
        timestampStr = doc.getString(YcIndexFields.TIMESTAMP);
        timestamp = ZonedDateTime.parse(timestampStr);
        resourceId = doc.get(YcIndexFields.RESOURCE_ID, NonEmptyValidator.TRIMMED);
        name = doc.getString(YcIndexFields.NAME);
        service = doc.get(YcIndexFields.SERVICE, NonEmptyValidator.TRIMMED);
        permission = doc.get(YcIndexFields.PERMISSION, NonEmptyValidator.TRIMMED);
        cloudId = doc.get(YcIndexFields.CLOUD_ID, NonEmptyValidator.TRIMMED);
        folderId = doc.getString(YcIndexFields.FOLDER_ID);

        deletedTsStr = doc.getString(YcIndexFields.DELETED, "");

        String reindexTimestampStr = doc.getString(YcIndexFields.REINDEX_TIMESTAMP, "");
        if (reindexTimestampStr != null && !reindexTimestampStr.isEmpty()) {
            this.reindexTimestamp = ZonedDateTime.parse(reindexTimestampStr);
        } else {
            this.reindexTimestamp = null;
        }

        JsonList resourcePathList = doc.getListOrNull("resource_path");
        if (resourcePathList == null) {
            this.resourcePath = null;
            this.resourcePathJson = null;
        } else {
            List<Map.Entry<String, String>> resourcePath = new ArrayList<>(resourcePathList.size());
            for (JsonObject pathObj: resourcePathList) {
                JsonMap resourcePathItemMap = pathObj.asMap();
                String resourceType = resourcePathItemMap.getString(YcIndexFields.RESOURCE_TYPE);
                String resourceId = resourcePathItemMap.getString(YcIndexFields.RESOURCE_ID);
                resourcePath.add(new AbstractMap.SimpleEntry<>(resourceType, resourceId));
            }

            this.resourcePath = Collections.unmodifiableList(resourcePath);
            this.resourcePathJson = resourcePathList;
        }

        Map<String, JsonObject> attributes = doc.getMapOrNull(YcIndexFields.ATTRIBUTES);
        if (attributes == null) {
            attributes = Collections.emptyMap();
        }

        this.attributes = attributes;
        timestampMicros = ChronoUnit.MICROS.between(Instant.EPOCH, timestamp.toInstant());
        if (reindexTimestamp != null) {
            reindexTimestampMicros = ChronoUnit.MICROS.between(Instant.EPOCH, reindexTimestamp.toInstant());
        } else {
            reindexTimestampMicros = -1;
        }

        if (!deletedTsStr.isEmpty()) {
            ZonedDateTime deleteTs = ZonedDateTime.parse(deletedTsStr);
            deleted = true;
            deleteTsMicros = ChronoUnit.MICROS.between(Instant.EPOCH, deleteTs.toInstant());
        } else {
            deleteTsMicros = -1L;
            deleted = false;
        }

        this.transferTs = transferTs;
    }

    @Override
    public void writeAttributeFields(final JsonWriterBase writer)
        throws IOException
    {
        writer.key(YcSearchFields.YC_RESOURCE_TYPE.stored());
        writer.value(resourceType);
        writer.key(YcSearchFields.YC_TIMESTAMP_STR.stored());
        writer.value(timestampStr);
        writer.key(YcSearchFields.YC_TIMESTAMP.stored());
        writer.value(timestampMicros);
        writer.key(YcSearchFields.YC_RESOURCE_ID.stored());
        writer.value(resourceId);
        writer.key(YcSearchFields.YC_NAME.stored());
        writer.value(name);
        writer.key(YcSearchFields.YC_SERVICE.stored());
        writer.value(service);
        writer.key(YcSearchFields.YC_CLOUD_ID.stored());
        writer.value(cloudId);
        writer.key(YcSearchFields.YC_FOLDER_ID.stored());
        writer.value(folderId);
        writer.key(YcSearchFields.YC_PERMISSION.stored());
        writer.value(permission);
        writer.key(YcSearchFields.YC_TRANSFER_TS.stored());
        writer.value(transferTs);
        writer.key(YcSearchFields.YC_DELETED_TIME.stored());
        writer.value(JsonNull.INSTANCE);
        writer.key(YcSearchFields.YC_SCHEMA_VERSION.stored());
        writer.value(2);
        writer.key(YcSearchFields.YC_MARKED_STALE_TS.stored());
        writer.value(JsonNull.INSTANCE);

        if (reindexTimestamp != null) {
            writer.key(YcSearchFields.YC_REINDEX_TIMESTAMP.stored());
            writer.value(reindexTimestampMicros);
        }

        if (resourcePath == null) {
            writer.key(YcSearchFields.YC_RESOURCE_PATH.stored());
            writer.value(JsonNull.INSTANCE);
        } else {
            writer.key(YcSearchFields.YC_RESOURCE_PATH.stored());
            writer.value(JsonType.NORMAL.toString(resourcePathJson));
        }
    }

    @Override
    public void writeMainFields(final JsonWriterBase writer)
        throws IOException
    {
        writer.key(YcSearchFields.YC_RESOURCE_TYPE.stored());
        writer.value(resourceType);
        writer.key(YcSearchFields.YC_TIMESTAMP_STR.stored());
        writer.value(timestampStr);
        writer.key(YcSearchFields.YC_TIMESTAMP.stored());
        writer.value(timestampMicros);
        writer.key(YcSearchFields.YC_RESOURCE_ID.stored());
        writer.value(resourceId);
        writer.key(YcSearchFields.YC_NAME.stored());
        writer.value(name);
        writer.key(YcSearchFields.YC_SERVICE.stored());
        writer.value(service);
        writer.key(YcSearchFields.YC_CLOUD_ID.stored());
        writer.value(cloudId);
        writer.key(YcSearchFields.YC_FOLDER_ID.stored());
        writer.value(folderId);
        writer.key(YcSearchFields.YC_PERMISSION.stored());
        writer.value(permission);
        writer.key(YcSearchFields.YC_TRANSFER_TS.stored());
        writer.value(transferTs);
        writer.key(YcSearchFields.YC_DELETED_TIME.stored());
        writer.value(JsonNull.INSTANCE);
        writer.key(YcSearchFields.YC_SCHEMA_VERSION.stored());
        writer.value(2);
        writer.key(YcSearchFields.YC_MARKED_STALE_TS.stored());
        writer.value(JsonNull.INSTANCE);

        if (resourcePath == null) {
            writer.key(YcSearchFields.YC_RESOURCE_PATH.stored());
            writer.value(JsonNull.INSTANCE);
        } else {
            writer.key(YcSearchFields.YC_RESOURCE_PATH.stored());
            writer.value(JsonType.NORMAL.toString(resourcePathJson));
        }

        if (reindexTimestamp != null) {
            writer.key(YcSearchFields.YC_REINDEX_TIMESTAMP.stored());
            writer.value(reindexTimestampMicros);
        }
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writeMainFields(writer);
    }

    @Override
    public String resourceType() {
        return resourceType;
    }

    @Override
    public long timestampMicros() {
        return timestampMicros;
    }

    @Override
    public ZonedDateTime timestamp() {
        return timestamp;
    }

    @Override
    public String timestampStr() {
        return timestampStr;
    }

    @Override
    public String resourceId() {
        return resourceId;
    }

    public String name() {
        return name;
    }

    @Override
    public String service() {
        return service;
    }

    @Override
    public String permission() {
        return permission;
    }

    @Override
    public String cloudId() {
        return cloudId;
    }

    @Override
    public String folderId() {
        return folderId;
    }

    public String deletedTsStr() {
        return deletedTsStr;
    }

    @Override
    public long deleteTsMicros() {
        return deleteTsMicros;
    }

    @Override
    public boolean deleted() {
        return deleted;
    }

    @Override
    public ZonedDateTime reindexTimestamp() {
        return reindexTimestamp;
    }

    @Override
    public Map<String, JsonObject> attributes() {
        return attributes;
    }

    public List<Map.Entry<String, String>> resourcePath() {
        return resourcePath;
    }

    public long transferTs() {
        return transferTs;
    }

    @Override
    public boolean reindex() {
        return reindexTimestamp != null;
    }
}
