package ru.yandex.search.yc;

import java.io.IOException;
import java.util.Map;

import org.apache.http.HttpHost;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.ProducerClientConfig;
import ru.yandex.collection.Pattern;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.search.yc.config.ImmutableYcSearchIndexerConfig;

public class YcIndexer
    extends UniversalSearchProxy<ImmutableYcSearchIndexerConfig>
{
    private final FieldGroupingStater serviceStater;
    private final FieldGroupingStater marketplaceProjectStater;
    private final ReindexWarden reindexWarden;
    private final ProducerClient marketProducerClient;
    private final HttpHost marketProducerHost;

    public YcIndexer(final ImmutableYcSearchIndexerConfig config)
        throws IOException, ConfigException
    {
        super(config);

        register(new Pattern<>("/api/yc/index", true), new CommonIndexHandler(this));
        register(new Pattern<>("/api/yc/marketplace/index", true), new YcMarketplaceIndexer(this));

        if (config.indexationTarget() == IndexationTargetFactory.BACKEND) {
            if (config.serviceStatConfig() != null) {
                serviceStater = new FieldGroupingStater(
                    this,
                    config.serviceStatConfig(),
                    new HttpHost(
                        config.producerClientConfig().host().getHostName(),
                        config.producerClientConfig().host().getPort() - 1),
                        "yc_service");
                registerStater(serviceStater);
                logger().info("Marketplace host " + config.marketStatConfig().host());

                marketplaceProjectStater = new FieldGroupingStater(
                    this,
                    config.marketStatConfig(),
                    new HttpHost(
                        config.marketStatConfig().host().getHostName(),
                        config.marketStatConfig().host().getPort() - 1),
                        "yc_project");
                registerStater(marketplaceProjectStater);
            } else {
                serviceStater = null;
                marketplaceProjectStater = null;
            }

            reindexWarden = new ReindexWarden(
                this,
                config.serviceStatConfig(),
                config.producerClientConfig().host(),
                new HttpHost(
                    config.producerClientConfig().host().getHostName(),
                    config.producerClientConfig().host().getPort() - 1));
            registerStater(reindexWarden);
        } else {
            serviceStater = null;
            marketplaceProjectStater = null;
            reindexWarden = null;
        }

        ProducerClientConfig marketProducerClientConfig =
                config.marketProducerClientConfig();

        marketProducerClient =
            registerClient(
                "Marketplace-Producer",
                new ProducerClient(
                    reactor,
                    config.producerClientConfig(),
                    searchMap()),
                config.producerClientConfig());
        marketProducerHost = marketProducerClientConfig.host();
    }

    @Override
    public void start() throws IOException {
        super.start();

        if (serviceStater != null) {
            serviceStater.start();
        }

        if (marketplaceProjectStater != null) {
            marketplaceProjectStater.start();
        }

        if (reindexWarden != null) {
            reindexWarden.start();
        }
    }


    @Override
    public Map<String, Object> status(boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("stale_stat", reindexWarden.status(verbose));
        return status;
    }

    public ProducerClient marketProducerClient() {
        return marketProducerClient;
    }

    public HttpHost marketProducerHost() {
        return marketProducerHost;
    }
}
