package ru.yandex.search.yc;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dispatcher.producer.SearchMap;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class YcMarketplaceIndexer implements ProxyRequestHandler {
    private static final File SCHEMA_FILE =
        new File(System.getenv().getOrDefault("LOGS_DIR","/logs") + "/last_marketplace_schema");
    private static final File DATA_FILE =
        new File(System.getenv().getOrDefault("LOGS_DIR","/logs") + "/last_marketplace_data");

    private static final StringBody EMPTY_BODY =
        new StringBody("", ContentType.TEXT_PLAIN);
    private static final boolean PRINT_MARKETPLACE_DATA =
        Boolean.parseBoolean(
            System.getenv().getOrDefault("PRINT_MARKETPLACE_DATA", "false"));

    private final YcIndexer ycIndexer;
    private ConcurrentHashMap<String, String> lastSchema = new ConcurrentHashMap<>();
    private ConcurrentHashMap<String, Long> lastTses = new ConcurrentHashMap<>();
    private final TimeFrameQueue<Integer> numdocs;

    public YcMarketplaceIndexer(final YcIndexer ycIndexer) {
        this.ycIndexer = ycIndexer;
        numdocs = new TimeFrameQueue<>(ycIndexer.config().metricsTimeFrame());
        ycIndexer.registerStater(
            new PassiveStaterAdapter<>(
                numdocs,
                    new NamedStatsAggregatorFactory<>(
                        "indexer-marketplace-numdocs_ammm",
                        IntegralSumAggregatorFactory.INSTANCE)));
        try {
            JsonMap schemas = TypesafeValueContentHandler.parse(new FileReader(SCHEMA_FILE, StandardCharsets.UTF_8)).asMap();
            for (Map.Entry<String, JsonObject> schemaEntry: schemas.entrySet()) {
                JsonMap schemaMap = schemaEntry.getValue().asMap();
                Long ts = schemaMap.getLong("ts");
                String data = schemaMap.getString("schema");
                lastSchema.put(schemaEntry.getKey(), data);
                lastTses.put(schemaEntry.getKey(), ts);
            }
        } catch (IOException | JsonException | NumberFormatException ioe) {
            ycIndexer.logger().info("Failed to load last schema");
            ioe.printStackTrace();
        }
    }

//    @Override
//    public HttpAsyncRequestConsumer<List<JsonMap>> processRequest(
//        final HttpRequest request,
//        final HttpContext context)
//        throws HttpException, IOException
//    {
//        if (!(request instanceof HttpEntityEnclosingRequest)) {
//            throw new BadRequestException("We accept post requests");
//        }
//
//        return new JsonStreamAsyncConsumer<>(
//            ((HttpEntityEnclosingRequest) request).getEntity(),
//            JsonObject::asMap,
//            ArrayList::new,
//            StringCollectorsFactory.INSTANCE,
//            BasicContainerFactory.INSTANCE);
//    }

    private Map.Entry<String, Map<String, MarketplaceFieldType>> generateSchema(final JsonMap root, final String project) throws JsonException {
        Map<String, String> schema = new LinkedHashMap<>();
        for (JsonObject docObj: root.getList("documents")) {
            for (JsonObject fieldObj: docObj.asMap().getList("fields")) {
                JsonMap field = fieldObj.asMap();
                schema.put(field.getString("name"), field.getString("indexType"));
            }
        }

        Map<String, MarketplaceFieldType> types = new LinkedHashMap<>();
        EnumParser<MarketplaceFieldType> fieldTypeParser = new EnumParser<>(MarketplaceFieldType.class);
        StringBuilder schemaSb = new StringBuilder();
        for (Map.Entry<String, String> entry: schema.entrySet()) {
            MarketplaceFieldType fieldType = fieldTypeParser.apply(entry.getValue());
            schemaSb.append(fieldType.generateField(project, entry.getKey()));
            schemaSb.append("\n");
            types.put(entry.getKey(), fieldType);
        }

        schemaSb.append(MarketplaceFieldType.generateIntField(project, "categories_map") +
            "store = true\n" +
            "index = false\n" +
            "\n" +
            MarketplaceFieldType.generateIntField(project, "data") +
            "store = true\n" +
            "index = false\n" +
            "\n" +
            MarketplaceFieldType.generateIntField(project, "logbroker_ts") +
            "tokenizer = keyword\n" +
            "store = true\n" +
            "prefixed = true\n" +
            "attribute = true\n" +
            "analyze = true\n" +
            "index_alias = " + MarketplaceFieldType.generateName(project, "yc_mkpl_int", "logbroker_ts_p") +
            "\n\n" +
            MarketplaceFieldType.generateIntField(project, "logbroker_ts_p") +
            "tokenizer = keyword\n" +
            "attribute = true\n" +
            "analyze = true\n" +
            "prefixed = true\n\n");

        return new AbstractMap.SimpleEntry<>(schemaSb.toString(), types);
    }

    protected void writeDoc(
        final JsonWriter writer,
        final JsonMap doc,
        final String project,
        final Map<String, MarketplaceFieldType> schema,
        final long ts)
        throws JsonException, IOException, BadRequestException
    {
        writer.startObject();

        JsonMap data = new JsonMap(BasicContainerFactory.INSTANCE);

        String language = null;

        String marketplaceId = doc.getString("id");
        data.put("id", doc.get("id"));
        for (JsonObject fieldObj: doc.getList("fields")) {
            JsonMap field = fieldObj.asMap();

            String name = field.getString("name");
            JsonObject valueObj = fieldObj.get("value");
            if ("language".equalsIgnoreCase(name)) {
                language = valueObj.asString();
            }

            if (valueObj.type() == JsonObject.Type.MAP) {
                throw new BadRequestException("field value is object, not supported " + name + " " + marketplaceId);
            }

            data.put(field.getString("name"), field.get("value"));

            MarketplaceFieldType fieldType = schema.get(name);
            if (fieldType == null) {
                throw new BadRequestException("Unknown schema for " + name);
            }

            fieldType.writeIndex(project, name, writer, valueObj);
        }

        String id = "yc_marketplace_doc_";
        if (project.isEmpty()) {
            id += marketplaceId;
        } else {
            id += project + "_" + marketplaceId;
        }

        if (language != null) {
            id += "_" + language;
        }
        writer.key("id");
        writer.value(id);

        writer.key("yc_project");
        writer.value(project.equals("") ? "default" : project);

        writer.key(MarketplaceFieldType.generateIntFieldName(project, "logbroker_ts"));
        writer.value(ts);
        writer.key(MarketplaceFieldType.generateIntFieldName(project, "data"));
        writer.value(JsonType.NORMAL.toString(data));
        writer.endObject();
    }


    private void handleProject(
        final FutureCallback<Object> callback,
        final ProxySession session,
        final String project,
        final JsonMap root,
        final long maxTs)
        throws HttpException, IOException, JsonException
    {
        Map.Entry<String, Map<String, MarketplaceFieldType>> schemaResult = generateSchema(root, project);
        String schema = schemaResult.getKey();
        synchronized (this) {
            String currentSchema = lastSchema.getOrDefault(project, "default");
            long ts = lastTses.getOrDefault(project, -1L);
            if (!schema.equals(currentSchema) && maxTs > ts) {
                lastSchema.put(project, schema);
                lastTses.put(project, maxTs);

                session.logger().info("Updating schema");
                if (PRINT_MARKETPLACE_DATA) {
                    synchronized (this) {
                        try (JsonWriter jsw = JsonType.HUMAN_READABLE.create(
                            new FileWriter(SCHEMA_FILE)))
                        {
                            jsw.startObject();
                            for (Map.Entry<String, String> entry: lastSchema.entrySet()) {
                                jsw.key(entry.getKey());
                                jsw.startObject();
                                jsw.key("ts");
                                jsw.value(lastTses.getOrDefault(entry.getKey(), -1L));
                                jsw.key("schema");
                                jsw.value(schema);
                                jsw.endObject();
                            }
                            jsw.endObject();
                        }
                    }

                    synchronized (this) {
                        try (FileWriter writer = new FileWriter(DATA_FILE)) {
                            writer.write(JsonType.HUMAN_READABLE.toString(root));
                        }
                    }
                }
            }
        }

        JsonList docs = root.getList("documents");
        session.logger().info("Updating marketplace docs " + docs.size() + " ts " + maxTs);

        session.logger().info("Working with service " + YcConstants.YC_MARKETPLACE_QUEUE);

        Prefix prefix = new StringPrefix(MarketplaceFieldType.prefix(project));
        StringBuilderWriter updateSbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(updateSbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(MarketplaceFieldType.prefix(project));
            writer.key("docs");
            writer.startArray();
            for (JsonObject docObj : docs) {
                writeDoc(writer, docObj.asMap(), project, schemaResult.getValue(), maxTs);
                numdocs.accept(1);
            }
            writer.endArray();
            writer.endObject();
        }

        ProducerClient client = ycIndexer.producerClient().adjust(session.context());
        QueryConstructor updateRequest = new QueryConstructor("/modify?marketplace");
        updateRequest.append("ts", maxTs);
        updateRequest.append("prefix", prefix.toStringFast());
        updateRequest.append("service", YcConstants.YC_MARKETPLACE_QUEUE);
        updateRequest.append("docs", docs.size());

        String shard = String.valueOf(prefix.hash() % SearchMap.SHARDS_COUNT);

        StringBuilder cleanupText = new StringBuilder();
        cleanupText.append(MarketplaceFieldType.generateIntFieldName(project, "logbroker_ts_p"));
        cleanupText.append(":[0 TO ");
        cleanupText.append(maxTs - 1);
        cleanupText.append("]");

        QueryConstructor cleanupRequest = new QueryConstructor("/delete?marketplace");
        cleanupRequest.append("text", cleanupText.toString());
        cleanupRequest.append("ts", maxTs);
        cleanupRequest.append("prefix", prefix.toStringFast());
        cleanupRequest.append("service", YcConstants.YC_MARKETPLACE_QUEUE);

        String schemaUpdateUri = "/schema_update?";
        if (!project.isEmpty()) {
            schemaUpdateUri += "schema-name=yc_marketplace_" + project;
        }

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype("mixed");
        builder.addPart(
            FormBodyPartBuilder.create()
                .setBody(
                    new StringBody(schema, ContentType.APPLICATION_JSON))
                .setName("schema.json")
                .addField(YandexHeaders.ZOO_SHARD_ID, shard)
                .addField(YandexHeaders.SERVICE, YcConstants.YC_MARKETPLACE_QUEUE)
                .addField(YandexHeaders.URI, schemaUpdateUri)
                .build());
        builder.addPart(
            FormBodyPartBuilder.create()
                .setBody(
                    new StringBody(updateSbw.toString(), ContentType.APPLICATION_JSON))
                .addField(YandexHeaders.ZOO_SHARD_ID, shard)
                .addField(YandexHeaders.SERVICE, YcConstants.YC_MARKETPLACE_QUEUE)
                .addField(YandexHeaders.URI, updateRequest.toString())
                .setName("update.json")
                .build());
        builder.addPart(
            FormBodyPartBuilder.create()
                .setBody(EMPTY_BODY)
                .addField(YandexHeaders.ZOO_HTTP_METHOD, "GET")
                .addField(YandexHeaders.ZOO_SHARD_ID, shard)
                .addField(YandexHeaders.SERVICE, YcConstants.YC_MARKETPLACE_QUEUE)
                .addField(YandexHeaders.URI, cleanupRequest.toString())
                .setName("cleanup.json")
                .build());

        QueryConstructor notifyQc = new QueryConstructor("/notify?marketplace_update");
        notifyQc.append("project", project);
        notifyQc.append("ts", maxTs);
        notifyQc.append("service", YcConstants.YC_MARKETPLACE_QUEUE);
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
            notifyQc.toString(),
                builder.build());
        //generator.addHeader(YandexHeaders.SERVICE, YcConstants.YC_QUEUE);
        //generator.addHeader(YandexHeaders.ZOO_SHARD_ID, shard);

        client.execute(
            ycIndexer.marketProducerHost(),
            generator,
            session.listener().adjustContextGenerator(
                client.httpClientContextGenerator()),
            callback);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        HttpEntity entity = ((HttpEntityEnclosingRequest) session.request()).getEntity();
        try {
            List<JsonObject> result =
                parseSingleMessage(CharsetUtils.content(entity), entity.getContentLength());
            handle(result, session);
        } catch (JsonException e) {
            throw new BadRequestException(e);
        }
    }

    private List<JsonObject> parseSingleMessage(
        final Reader reader,
        final long expectedSize)
        throws IOException, JsonException
    {
        JsonList parseResult = new JsonList(BasicContainerFactory.INSTANCE);
        JsonParser parser = new JsonParser(
            new StackContentHandler(
                new TypesafeValueContentHandler(
                    parseResult,
                    StringCollectorsFactory.INSTANCE.apply(expectedSize),
                    BasicContainerFactory.INSTANCE)),
            true);
        parser.parse(reader);
        return parseResult;
    }

    public void handle(
        final List<JsonObject> lbDocs,
        final ProxySession session)
        throws HttpException, IOException
    {
        Callback callback = new Callback(session);

        if (lbDocs.size() <= 0) {
            session.response(HttpStatus.SC_OK);
            session.logger().warning("No docs for marketplace");
            return;
        }

        try {
            Map<String, JsonMap> roots = new LinkedHashMap<>();
            Map<String, Long> maxTss = new LinkedHashMap<>();
            for (JsonObject mapObj: lbDocs) {
                JsonMap map = mapObj.asMap();
                long ts = map.getLong("timestamp");
                String project = map.getString("project", "default");
                session.logger().info("Marketplace lbdoc "  + ts + " " + JsonType.NORMAL.toString(map));
                if (ts >= maxTss.getOrDefault(project, Long.MIN_VALUE)) {
                    maxTss.put(project, ts);
                    roots.put(project, map);
                }
            }

            MultiFutureCallback<Object> mfcb = new MultiFutureCallback<>(callback);
            for (Map.Entry<String, JsonMap> entry: roots.entrySet()) {
                String project = entry.getKey();
                handleProject(
                    mfcb.newCallback(),
                    session,
                    project,
                    entry.getValue(),
                    maxTss.get(entry.getKey()));
            }
            mfcb.done();
        } catch (JsonException je) {
            callback.failed(je);
        }
    }

    private static class Callback extends AbstractProxySessionCallback<Object> {
        public Callback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Object o) {
            session.response(HttpStatus.SC_OK);
        }
    }
}

