package ru.yandex.search.yc;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.nio.charset.UnsupportedCharsetException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HTTP;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeTokenStream;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.search.yc.schema.SchemaValidationException;
import ru.yandex.search.yc.schema.YcSearchIndexSelectSchemaParser;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.yc.search.YcMalformedSearchDocsFields;

public class YcSearchIndexer implements ProxyRequestHandler {
    private final YcIndexer ycIndexer;
    private final YcSearchIndexSelectSchemaParser parser;
    private final PrefixedLogger incomingLogger;
    private final PrefixedLogger outgoingLogger;

    private final TimeFrameQueue<Integer> updates;
    private final TimeFrameQueue<Integer> deletes;
    private final TimeFrameQueue<Integer> skipped;
    private final FieldGroupingElementStater acceptedByService;
    private final FieldGroupingElementStater failedByService;
    private final FieldGroupingElementStater schemaV1ByService;
    private final FieldGroupingElementStater schemaV2ByService;

    private final List<HttpHost> cloudHosts;
    private final IndexationTarget indexationTarget;

    public YcSearchIndexer(final YcIndexer ycIndexer) throws IOException {
        this.ycIndexer = ycIndexer;
        this.parser = new YcSearchIndexSelectSchemaParser(this);

        String cloudProducerHostsStr = System.getenv().getOrDefault("YC_CLOUD_PRODUCER_HOSTS", "");
        if (!cloudProducerHostsStr.isEmpty()) {
            cloudHosts =
                new CollectionParser<>(HttpHostParser.INSTANCE, ArrayList::new).apply(cloudProducerHostsStr);
        } else{
            cloudHosts = Collections.emptyList();
        }
        incomingLogger =
            ycIndexer.config().loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/incoming/docs")));

        outgoingLogger =
            ycIndexer.config().loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/outgoing/docs")));

        updates = new TimeFrameQueue<>(ycIndexer.config().metricsTimeFrame());
        ycIndexer.registerStater(
            new PassiveStaterAdapter<>(
                updates,
                new NamedStatsAggregatorFactory<>(
                    "indexer-updates_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        deletes = new TimeFrameQueue<>(ycIndexer.config().metricsTimeFrame());
        ycIndexer.registerStater(
            new PassiveStaterAdapter<>(
                deletes,
                new NamedStatsAggregatorFactory<>(
                    "indexer-deletes_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        skipped = new TimeFrameQueue<>(ycIndexer.config().metricsTimeFrame());
        ycIndexer.registerStater(
            new PassiveStaterAdapter<>(
                skipped,
                new NamedStatsAggregatorFactory<>(
                    "indexer-skipped_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        acceptedByService = new FieldGroupingElementStater(
            ycIndexer.config().metricsTimeFrame(),
            "search_indexation_parsed_documents_");
        ycIndexer.registerStater(acceptedByService);
        failedByService = new FieldGroupingElementStater(
            ycIndexer.config().metricsTimeFrame(),
            "search_indexation_parse_failed_documents_");
        ycIndexer.registerStater(failedByService);
        schemaV1ByService = new FieldGroupingElementStater(
            ycIndexer.config().metricsTimeFrame(),
            "search_indexation_parse_schema_v1_");
        ycIndexer.registerStater(schemaV1ByService);
        schemaV2ByService = new FieldGroupingElementStater(
            ycIndexer.config().metricsTimeFrame(),
            "search_indexation_parse_schema_v2_");
        ycIndexer.registerStater(schemaV2ByService);

        this.indexationTarget = ycIndexer.config().indexationTarget().create(this);
    }

    public FieldGroupingElementStater schemaV1ByService() {
        return schemaV1ByService;
    }

    public FieldGroupingElementStater schemaV2ByService() {
        return schemaV2ByService;
    }

    public YcIndexer server() {
        return ycIndexer;
    }

    public PrefixedLogger outgoingLogger() {
        return outgoingLogger;
    }

    public TimeFrameQueue<Integer> updates() {
        return updates;
    }

    public TimeFrameQueue<Integer> deletes() {
        return deletes;
    }

    public TimeFrameQueue<Integer> skipped() {
        return skipped;
    }

    public FieldGroupingElementStater acceptedByService() {
        return acceptedByService;
    }

    public FieldGroupingElementStater failedByService() {
        return failedByService;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        final String contentTypeHeader =
            session.headers().getString(HTTP.CONTENT_TYPE, null);
        ContentType contentType = null;
        if (contentTypeHeader != null) {
            try {
                contentType = ContentType.parse(contentTypeHeader);
            } catch (org.apache.http.ParseException
                | UnsupportedCharsetException e)
            {
                throw new BadRequestException("Bad content-type header");
            }
        }

        final boolean multiPart;
        if (contentType != null
            && contentType.getMimeType().equals("multipart/mixed"))
        {
            multiPart = true;
        } else {
            multiPart = false;
        }

        try {
            if (multiPart) {
                handleMultipart(session, parseMultipart(session));
            } else {
                handleSingleMessage(session);
            }
        } catch (ServerException e) {
            throw e;
        } catch (Exception e) {
            throw new BadRequestException("Can't handle request", e);
        }
    }

    protected JsonObject jsonFunction(
        final String function,
        final String arg1)
    {
        return jsonFunction(function, new JsonString(arg1));
    }

    protected JsonObject jsonFunction(
        final String function,
        final JsonObject arg1,
        final String arg2)
    {
        return jsonFunction(function, arg1, new JsonString(arg2));
    }

    protected JsonObject jsonFunction(
        final String function,
        final JsonObject... args)
    {
        JsonMap func = new JsonMap(BasicContainerFactory.INSTANCE);
        func.put("function", new JsonString(function));
        JsonList list = new JsonList(BasicContainerFactory.INSTANCE);
        for (JsonObject arg: args) {
            list.add(arg);
        }
        func.put("args", list);
        return func;
    }

    private void indexMalformedMessage(
        final YcSearchIndexRequestContext context,
        final Exception e,
        final JsonMap doc,
        final int docNum,
        final String service,
        final FutureCallback<Object> callback)
        throws IOException, BadRequestException
    {
        String prefix = "0";
        QueryConstructor qc = new QueryConstructor("/modify?malformed_search_doc");
        qc.append("yc_service", service);
        qc.append("topic", context.topic());
        qc.append("offset", context.offset());
        qc.append("service", ycIndexer.config().malformedDocsService());
        qc.append("db", "malformed_search_docs");
        qc.append("prefix", prefix);

        StringBuilder docId = new StringBuilder("yc_malformed_doc_");
        docId.append(context.topic());
        docId.append("_");
        docId.append(context.partition());
        docId.append("_");
        docId.append(context.offset());
        docId.append("_");
        docId.append(context.seqNo());
        docId.append("_");
        docId.append(docNum);

        StringBuilderWriter postSbw = new StringBuilderWriter();
        String errorMessage;
        try (JsonWriter writer = JsonType.NORMAL.create(postSbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(prefix);
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            writer.key(YcMalformedSearchDocsFields.ID.stored());
            writer.value(docId.toString());
            writer.key(YcMalformedSearchDocsFields.YC_MF_TOPIC.stored());
            writer.value(context.topic());
            writer.key(YcMalformedSearchDocsFields.YC_MF_MESSAGE_CREATE_TIME.stored());
            writer.value(context.messageCreateTime());
            writer.key(YcMalformedSearchDocsFields.YC_MF_MESSAGE_WRITE_TIME.stored());
            writer.value(context.messageWriteTime());
            writer.key(YcMalformedSearchDocsFields.YC_MF_PARTITION.stored());
            writer.value(context.partition());
            writer.key(YcMalformedSearchDocsFields.YC_MF_SEQNO.stored());
            writer.value(context.seqNo());
            writer.key(YcMalformedSearchDocsFields.YC_MF_TOPIC_OFFSET.stored());
            writer.value(context.offset());
            writer.key(YcMalformedSearchDocsFields.YC_MF_SERVICE.stored());
            writer.value(service);
            writer.key(YcMalformedSearchDocsFields.YC_MF_MESSAGE_TRANSFER_TS.stored());
            writer.value(context.transferTs());
            writer.key(YcMalformedSearchDocsFields.YC_MF_MESSAGE_DATA.stored());
            writer.value(JsonType.NORMAL.toString(doc));
            writer.key(YcMalformedSearchDocsFields.YC_MF_MESSAGE_PARSE_ERROR.stored());
            if (e instanceof SchemaValidationException) {
                errorMessage = ((SchemaValidationException) e).validatorMessage();
            } else {
                StringWriter sw = new StringWriter();
                e.printStackTrace(new PrintWriter(sw));
                errorMessage = sw.toString();
            }

            writer.value(errorMessage);
            writer.endObject();
            writer.endArray();
            writer.endObject();
        }

        BasicAsyncRequestProducerGenerator post = new BasicAsyncRequestProducerGenerator(
            qc.toString(),
            postSbw.toString());

        post.addHeader(YandexHeaders.SERVICE, ycIndexer.config().malformedDocsService());

        context.session().logger().log(Level.WARNING,
            "Parse For Doc Failed " + JsonType.NORMAL.toString(doc),
            e);
        context.client().execute(
            ycIndexer.producerHost(),
            post,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            callback);
    }

    private void handle(
        final List<JsonObject> lbDocs,
        final YcSearchIndexRequestContext requestContext,
        final FutureCallback<Object> sessionCallback)
    {
        ProxySession session = requestContext.session();
        PrefixedLogger incLogger =
            incomingLogger.addPrefix((String) session.context().getAttribute(HttpServer.SESSION_ID));
        MultiFutureCallback<Object> mfcb =
            new MultiFutureCallback<>(sessionCallback);

        int docNum = -1;
        for (JsonObject data: lbDocs) {
            docNum++;
            if (requestContext.logIncomingDoc()) {
                incLogger.info(JsonType.NORMAL.toString(data));
            }

            YcDoc doc;
            String service = null;
            try {
                JsonMap root = data.asMap();
                service = root.getString(YcIndexFields.SERVICE);
//                String cloudId = root.getString(YcIndexFields.CLOUD_ID, "");
//                if (cloudId.isEmpty()) {
//                    throw new JsonException("No cloudId supplied")
//                }

                doc = parser.parse(session.logger(), root, requestContext.transferTs());
                indexationTarget.accept(requestContext, Collections.singletonList(doc), mfcb.newCallback());
            } catch (Exception je) {
                if (service != null) {
                    failedByService.getOrCreate(service).accept(1L);
                }

                switch (requestContext.badIndexMessageAction()) {
                    case SKIP:
                        session.logger().warning("Skipping message for service " + service  + " data "  + JsonType.NORMAL.toString(data));
                        continue;
                    case SAVE_IN_INDEX:
                        try {
                            indexMalformedMessage(requestContext, je, data.asMap(), docNum, service, mfcb.newCallback());
                        } catch (Exception e) {
                            sessionCallback.failed(
                                new BadRequestException(
                                    "Failed to save to index malformed  json " + JsonType.NORMAL.toString(data), e)
                            );
                            return;
                        }

                        continue;
                    case HALT:
                    default:
                        sessionCallback.failed(new BadRequestException(
                            "Invalid json " + JsonType.NORMAL.toString(data), je));
                        return;
                }
            }
        }
        mfcb.done();

    }

    private void launchToCloud(
        final List<JsonObject> lbDocs,
        final YcSearchIndexRequestContext requestContext,
        final FutureCallback<Object> callback)
    {
        StringBuilder sb = new StringBuilder();
        for (JsonObject doc: lbDocs) {
            sb.append(JsonType.NORMAL.toString(doc));
            sb.append("\n");
        }
        CgiParams params = requestContext.session().params();
        QueryConstructor qc = new QueryConstructor("/api/yc/index?");
        try {
            qc.copyIfPresent(params, "topic");
            qc.copyIfPresent(params, "partition");
            qc.copyIfPresent(params, "offset");
            qc.copyIfPresent(params, "seqNo");
            qc.copyIfPresent(params, "message-create-time");
            qc.copyIfPresent(params, "message-write-time");
            qc.copyIfPresent(params, "transfer_ts");
        } catch (BadRequestException bre) {
            callback.failed(bre);
            return;
        }

        BasicAsyncRequestProducerGenerator cloudPost = new BasicAsyncRequestProducerGenerator(
            qc.toString(),
            sb.toString(),
            StandardCharsets.UTF_8);

        requestContext.session().logger().info("Sending request to cloud " + this.cloudHosts);
        List<HttpHost> cloudHosts = new ArrayList<>(this.cloudHosts);
        Collections.shuffle(cloudHosts);
        MultiFutureCallback<Object> mfcb = new MultiFutureCallback<>(callback);
        for (HttpHost host: cloudHosts) {
            requestContext.client().execute(
                Collections.singletonList(host),
                cloudPost,
                System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(10),
                2500L,
                BasicAsyncResponseConsumerFactory.ANY_GOOD,
                requestContext.session().listener().adjustContextGenerator(
                    requestContext.client().httpClientContextGenerator()),
                mfcb.newCallback());
        }
        mfcb.done();
    }

    private List<JsonObject> parseSingleMessage(
        final Reader reader,
        final long expectedSize)
        throws IOException, JsonException
    {
        JsonList parseResult = new JsonList(BasicContainerFactory.INSTANCE);
        JsonParser parser = new JsonParser(
            new StackContentHandler(
                new TypesafeValueContentHandler(
                    parseResult,
                    StringCollectorsFactory.INSTANCE.apply(expectedSize),
                    BasicContainerFactory.INSTANCE)),
            true);
        parser.parse(reader);
        return parseResult;
    }

    private void handleSingleMessage(final ProxySession session)
        throws HttpException, IOException, JsonException
    {
        HttpEntity entity = ((HttpEntityEnclosingRequest) session.request()).getEntity();
        final List<JsonObject> result =
            parseSingleMessage(CharsetUtils.content(entity), entity.getContentLength());

        YcSearchIndexRequestContext requestContext
            = new YcSearchIndexRequestContext(this, session);


        if (!cloudHosts.isEmpty()) {
            launchToCloud(
                result,
                requestContext,
                new FilterFutureCallback<>(new SessionCallback(session)) {
                    @Override
                    public void completed(final Object resultObj) {
                        requestContext.session().logger().info("Cloud request completed");
                        handle(result, requestContext, callback);
                    }
                });
        } else {
            handle(result, requestContext, new SessionCallback(session));
        }

    }

    private void handleMultipart(
        final ProxySession session,
        final List<Map.Entry<CgiParams, List<JsonObject>>> parts)
        throws HttpException
    {
        SessionCallback sessionCallback = new SessionCallback(session);
        MultiFutureCallback<Object> mfcb = new MultiFutureCallback<>(sessionCallback);
        for (Map.Entry<CgiParams, List<JsonObject>> entry: parts) {
            YcSearchIndexRequestContext requestContext
                = new YcSearchIndexRequestContext(this, session, entry.getKey());

            if (!cloudHosts.isEmpty()) {
                launchToCloud(
                    entry.getValue(),
                    requestContext,
                    new FilterFutureCallback<>(mfcb.newCallback()) {
                        @Override
                        public void completed(final Object result) {
                            requestContext.session().logger().info("Cloud request completed");
                            handle(entry.getValue(), requestContext, callback);
                        }
                    });
            } else {
                handle(entry.getValue(), requestContext, mfcb.newCallback());
            }
        }
        mfcb.done();
    }

    private List<Map.Entry<CgiParams, List<JsonObject>>> parseMultipart(
        final ProxySession session)
        throws HttpException
    {
        List<Map.Entry<CgiParams, List<JsonObject>>> parts = new ArrayList<>();
        MimeTokenStream stream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());

        try {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) session.request()).getEntity();
            String uri = null;
            stream.parseHeadless(
                entity.getContent(),
                entity.getContentType().getValue());
            EntityState state = stream.getState();
            while (state != EntityState.T_END_OF_STREAM) {
                switch (state) {
                    case T_FIELD:
                        Field field = stream.getField();
                        if (YandexHeaders.URI
                            .equals(field.getName()))
                        {
                            uri = field.getBody();
                        }
                        break;
                    case T_BODY:
                        int pos = 0;

                        List<JsonObject> messages = Collections.emptyList();
                        try (InputStream is = stream.getDecodedInputStream()) {
                            messages =
                                parseSingleMessage(
                                    new InputStreamReader(is),
                                    stream.getBodyDescriptor().getContentLength());
                        }
                        if (uri == null) {
                            throw new BadRequestException("Empty URI header for"
                                + " <" + parts.size() + "> multipart section");
                        }
                        UriParser uriParser = new UriParser(uri);
                        CgiParams cgiParams =
                            new CgiParams(uriParser.queryParser());
                        parts.add(new AbstractMap.SimpleEntry<>(cgiParams, messages));
                        uri = null;
                        break;
                    default:
                        break;
                }
                state = stream.next();
            }
        } catch (Throwable t) {
            throw new NotImplementedException(t);
        }
        return parts;
    }

//    @Override
//    public void handle(
//        final List<JsonObject> lbDocs,
//        final HttpAsyncExchange exchange,
//        final HttpContext context)
//        throws HttpException, IOException
//    {
//        ProxySession session = new BasicProxySession(ycIndexer, exchange, context);
//        YcSearchIndexRequestContext requestContext
//            = new YcSearchIndexRequestContext(this, session);
//
//        SessionCallback sessionCallback = new SessionCallback(session);
//        if (!cloudHosts.isEmpty()) {
//            launchToCloud(
//                lbDocs,
//                requestContext,
//                new FilterFutureCallback<>(sessionCallback) {
//                    @Override
//                    public void completed(final Object result) {
//                        requestContext.session().logger().info("Cloud request completed");
//                        handle(lbDocs, requestContext, sessionCallback);
//                    }
//                });
//        } else {
//            handle(lbDocs, requestContext, sessionCallback);
//        }
//    }

    private static final class SessionCallback extends AbstractProxySessionCallback<Object> {
        public SessionCallback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(Object o) {
            //session.logger().fine("Done");
            session.response(HttpStatus.SC_OK);
        }
    }

}
