package ru.yandex.search.yc.config;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.LinkedHashMap;
import java.util.Map;

import ru.yandex.client.producer.ProducerClientConfig;
import ru.yandex.client.producer.ProducerClientConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.search.proxy.universal.AbstractUniversalSearchProxyConfigBuilder;
import ru.yandex.search.yc.BadIndexMessageAction;
import ru.yandex.search.yc.IndexationTargetFactory;
import ru.yandex.search.yc.schema.YcSearchIndexDocParser;

public abstract class AbstractYcSearchIndexerConfigBuilder
    <T extends AbstractYcSearchIndexerConfigBuilder<T>>
     extends AbstractUniversalSearchProxyConfigBuilder<T>
     implements YcSearchIndexerConfig
{
    private BadIndexMessageAction badIndexMessageAction;
    private String malformedDocsService;
    private IndexationTargetFactory indexationTarget;
    private Map<String, YcSearchIndexDocParser> schemas;
    private ProducerClientConfigBuilder marketProducerClientConfig;
    private FieldGroupingElementStatersConfigBuilder serviceStatConfig;
    private FieldGroupingElementStatersConfigBuilder marketStatConfig;

    public AbstractYcSearchIndexerConfigBuilder(final YcSearchIndexerConfig config) {
        super(config);

        badIndexMessageAction(config.badIndexMessageAction());
        malformedDocsService(config.malformedDocsService());
        schemas(config.schemas());
        indexationTarget(config.indexationTarget());
        serviceStatConfig(config.serviceStatConfig());
        marketStatConfig(config.marketStatConfig());
    }

    public AbstractYcSearchIndexerConfigBuilder(
        final IniConfig config,
        final YcSearchIndexerConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        this.badIndexMessageAction =
            config.getEnum(
                BadIndexMessageAction.class,
                "bad-index-message-action",
                defaults.badIndexMessageAction());
        malformedDocsService = config.getString("malformed-docs-service", null);
        indexationTarget =
            config.getEnum(
                IndexationTargetFactory.class,
                "indexation-target",
                defaults.indexationTarget());

        IniConfig schemaSection = config.sectionOrNull("schema");
        if (schemaSection == null) {
            schemas = new LinkedHashMap<>(defaults.schemas());
        } else {
            Map<String, YcSearchIndexDocParser> map = new LinkedHashMap<>();
            for (Map.Entry<String, IniConfig> entry: schemaSection.sections().entrySet()) {
                IniConfig schemaConfig = entry.getValue();
                String className = schemaConfig.getString("parser", null);
                File schemaFile = schemaConfig.getInputFile("file");
                YcSearchIndexDocParser parser;
                try {
                    Class<?> configClass = Class.forName(className);
                    parser = (YcSearchIndexDocParser) configClass
                        .getConstructor(File.class)
                        .newInstance(schemaFile);
                    map.put(entry.getKey(), parser);
                } catch (ClassNotFoundException | NoSuchMethodException
                    | InstantiationException | IllegalAccessException
                    | IllegalArgumentException | InvocationTargetException e)
                {
                    throw new ConfigException(e);
                }
            }

            this.schemas = map;

//            marketProducerClientConfig = new ProducerClientConfigBuilder(
//                    config.section("market-producer"),
//                    defaults.marketProducerClientConfig());

            IniConfig marketProducerSection = config.sectionOrNull("market-producer");
            ProducerClientConfig marketProducerDefaults =
                    defaults.marketProducerClientConfig();
            if (marketProducerSection == null) {
                marketProducerClientConfig(marketProducerDefaults);
            } else if (marketProducerDefaults == null) {
                marketProducerClientConfig =
                        new ProducerClientConfigBuilder(marketProducerSection);
            } else {
                marketProducerClientConfig = new ProducerClientConfigBuilder(
                        marketProducerSection,
                        marketProducerDefaults);
            }

            IniConfig serviceStatSection = config.sectionOrNull("services-stat");
            if (serviceStatSection != null) {
                this.serviceStatConfig =
                    new FieldGroupingElementStatersConfigBuilder(serviceStatSection, defaults.serviceStatConfig());
            } else {
                this.serviceStatConfig = new FieldGroupingElementStatersConfigBuilder(defaults.serviceStatConfig());
            }

            IniConfig marketStatSection = config.sectionOrNull("marketplace-stat");
            if (marketStatSection != null) {
                this.marketStatConfig =
                        new FieldGroupingElementStatersConfigBuilder(marketStatSection, defaults.marketStatConfig());
            } else {
                this.marketStatConfig = new FieldGroupingElementStatersConfigBuilder(defaults.marketStatConfig());
            }
        }
    }

    @Override()
    public BadIndexMessageAction badIndexMessageAction() {
        return badIndexMessageAction;
    }

    public T badIndexMessageAction(final BadIndexMessageAction value) {
        this.badIndexMessageAction = value;
        return self();
    }

    @Override
    public String malformedDocsService() {
        return malformedDocsService;
    }

    public T malformedDocsService(final String malformedDocsService) {
        this.malformedDocsService = malformedDocsService;
        return self();
    }

    @Override
    public Map<String, YcSearchIndexDocParser> schemas() {
        return schemas;
    }

    public T schemas(final Map<String, YcSearchIndexDocParser> schemas) {
        this.schemas = new LinkedHashMap<>(schemas);
        return self();
    }

    @Override
    public IndexationTargetFactory indexationTarget() {
        return indexationTarget;
    }

    public T indexationTarget(final IndexationTargetFactory target) {
        this.indexationTarget = target;
        return self();
    }

    @Override
    public ProducerClientConfigBuilder marketProducerClientConfig() {
        return marketProducerClientConfig;
    }

    public T marketProducerClientConfig(final ProducerClientConfig marketProducerClientConfig) {
        this.marketProducerClientConfig = new ProducerClientConfigBuilder(marketProducerClientConfig);
        return self();
    }

    @Override
    public FieldGroupingElementStatersConfigBuilder serviceStatConfig() {
        return serviceStatConfig;
    }

    @Override
    public FieldGroupingElementStatersConfigBuilder marketStatConfig() {
        return marketStatConfig;
    }

    public T serviceStatConfig(final FieldGroupingElementStatersConfig config) {
        this.serviceStatConfig = new FieldGroupingElementStatersConfigBuilder(config);
        return self();
    }

    public T marketStatConfig(final FieldGroupingElementStatersConfig config) {
        this.marketStatConfig = new FieldGroupingElementStatersConfigBuilder(config);
        return self();
    }
}
