package ru.yandex.search.yc;

import java.io.File;
import java.nio.file.Files;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import ru.yandex.function.FalsePredicate;
import ru.yandex.function.GenericFunction;
import ru.yandex.function.TruePredicate;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.util.string.StringUtils;

public enum AllowedServicesParser
    implements GenericFunction<String, Predicate<String>, Exception>
{
    INSTANCE;

    private static final CollectionParser<String, Set<String>, Exception> SET_PARSER =
        new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);
    private static final EnumParser<PredicateFactory> PREDICATE_FACTORY_PARSER
        = new EnumParser<>(PredicateFactory.class);

    @Override
    public Predicate<String> apply(final String s) throws Exception {
        String trimmed = s.trim();
        int colon = trimmed.indexOf(':');

        if (colon < 0) {
            return PREDICATE_FACTORY_PARSER.apply(s).create("");
        } else {
            String name = trimmed.substring(0, colon);
            return PREDICATE_FACTORY_PARSER.apply(name).create(trimmed.substring(colon + 1));
        }
    }

    private enum PredicateFactory {
        WHITELIST {
            @Override
            Predicate<String> create(final String param) throws Exception {
                return new WhitelistPredicate(SET_PARSER.apply(param));
            }
        },
        WHITELIST_FILE {
            @Override
            Predicate<String> create(final String param) throws Exception {
                List<String> lines = Files.readAllLines(new File(param).toPath());
                Set<String> services =
                    lines.stream().filter(Predicate.not(String::isBlank)).collect(Collectors.toSet());
                return new WhitelistPredicate(services);
            }
        },
        BLACKLIST {
            @Override
            Predicate<String> create(final String param) throws Exception {
                return new BlacklistPredicate(SET_PARSER.apply(param));
            }
        },
        ALL {
            @Override
            Predicate<String> create(final String param) throws Exception {
                return TruePredicate.instance();
            }
        },
        NONE {
            @Override
            Predicate<String> create(final String param) throws Exception {
                return FalsePredicate.instance();
            }
        };

        abstract Predicate<String> create(final String param) throws Exception;

    }

    private static class WhitelistPredicate implements Predicate<String> {
        private final Set<String> services;
        private final String stringRepr;

        public WhitelistPredicate(final Set<String> services) {
            this.services = services;
            this.stringRepr = "whitelisted:" + StringUtils.join(services, ',');
        }

        @Override
        public boolean test(final String s) {
            return services.contains(s);
        }

        @Override
        public String toString() {
            return stringRepr;
        }
    }

    private static class BlacklistPredicate implements Predicate<String> {
        private final Set<String> services;
        private final String stringRepr;

        public BlacklistPredicate(final Set<String> services) {
            this.services = services;
            this.stringRepr = "blacklisted:" + StringUtils.join(services, ',');
        }

        @Override
        public boolean test(final String s) {
            return !services.contains(s);
        }

        @Override
        public String toString() {
            return stringRepr;
        }
    }
}
