package ru.yandex.search.yc;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpHeaders;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;

public class MultiCloudYCSearchContext extends TokenRequiredYcSearchContext {
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);
    private static final String FOLDER_ID_REFERER_REGEX =
        "^https://console(-preprod)?\\.cloud\\.yandex\\.ru/folders/([a-zA-Z0-9]+)";

    protected final ProxySession session;
    protected final JsonType jsonType;
    protected final AsyncClient client;
    protected final YcSearchProxy proxy;
    protected final Locale locale;

    protected final boolean allowLaggingHosts;
    protected final boolean debug;
    protected final int length;
    protected final int offset;

    protected final String request;
    protected final Set<String> cloudIds;
    protected final String folderId;
    protected final boolean highlight;

    public MultiCloudYCSearchContext(
        final YcSearchProxy proxy,
        final ProxySession session)
        throws BadRequestException
    {
        super(session);

        this.proxy = proxy;
        this.session = session;

        CgiParams params = session.params();

        List<String> cloudIds = params.getAll("cloud_id");

        if (cloudIds.isEmpty()) {
            this.cloudIds = params.get("cloud_ids", Collections.emptySet(), SET_PARSER);

//            if (this.cloudIds.isEmpty()) {
//                throw new BadRequestException("No cloud id supplied");
//            }
        } else {
            this.cloudIds = new LinkedHashSet<>(cloudIds);
        }

        this.folderId = getFolderId(params);

        jsonType = JsonTypeExtractor.NORMAL.extract(params);
        allowLaggingHosts =
            params.getBoolean("allow-lagging-hosts", true);

        client = proxy.searchClient().adjust(session.context());

        locale = params.getLocale("locale", Locale.ROOT);

        request = params.getString("request");
        debug = params.getBoolean("debug", false);
        length = params.get("length", 10, PositiveIntegerValidator.INSTANCE);
        offset = params.get("offset", 0, NonNegativeIntegerValidator.INSTANCE);
        highlight = params.getBoolean("highlight", true);
        session.logger().info("Highlight " + highlight);
    }

    private String getFolderId(CgiParams params) {
        String folderId = params.getOrNull("folder_id");
        if (folderId != null) {
            return folderId;
        }
        String referer = session.headers().getOrNull(HttpHeaders.REFERER);
        if (referer == null) {
            return null;
        }
        Pattern folderIdRefererPattern = Pattern.compile(FOLDER_ID_REFERER_REGEX);
        Matcher folderIdRefererMatcher = folderIdRefererPattern.matcher(referer);
        if (folderIdRefererMatcher.find()) {
            return folderIdRefererMatcher.group(2);
        }
        return null;
    }

    public ProxySession session() {
        return session;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public AsyncClient client() {
        return client;
    }

    public YcSearchProxy proxy() {
        return proxy;
    }

    public Locale locale() {
        return locale;
    }

    public boolean allowLaggingHosts() {
        return allowLaggingHosts;
    }

    public boolean debug() {
        return debug;
    }

    public int length() {
        return length;
    }

    public int offset() {
        return offset;
    }

    public String request() {
        return request;
    }

    public Set<String> cloudIds() {
        return cloudIds;
    }

    public boolean highlight() {
        return highlight;
    }

    public String folderId() {
        return folderId;
    }
}
