package ru.yandex.search.yc;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapRow;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.rules.pure.SearchRule;

public class SearchHandler implements ProxyRequestHandler {
    private final SearchRule<BasicYcSearchContext, YcSearchResult<BasicYcResultItem>> rule;
    private final YcSearchProxy proxy;

    public SearchHandler(final YcSearchProxy proxy) {
        this.proxy = proxy;
        this.rule = new YcPlainSearchRule<>();
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        MultiCloudYCSearchContext multiCloudContext =
            new MultiCloudYCSearchContext(proxy, session);

        YcSearchResultPrinter printer =
            new YcSearchResultPrinter(multiCloudContext);

        //https://st.yandex-team.ru/CLOUD-90875
        if (multiCloudContext.cloudIds().isEmpty()) {
            session.logger().info("Empty cloud ids");
            printer.completed(new YcSearchResult<>());
            return;
        }

        if (multiCloudContext.request().isEmpty()) {
            printer.completed(new YcSearchResult<>());
            return;
        }

        // now we have all shards on same node, so for now we could combine all
        // cloudIds in one request, for compatibility for future sharding - keep multifuture callback
        MultiFutureCallback<YcSearchResult<BasicYcResultItem>> mfcb =
            new MultiFutureCallback<>(
                new ConcatCallback(printer, multiCloudContext));

        Map<SearchMapShard, Set<String>> shardedClouds =
            new LinkedHashMap<>(multiCloudContext.cloudIds().size() << 1);

        SearchMapRow searchMapRow = proxy.searchMap().row(YcConstants.YC_QUEUE);
        for (String cloudId: multiCloudContext.cloudIds()) {
            StringPrefix prefix = new StringPrefix(cloudId);
            int shardId = (int) (prefix.hash() % SearchMap.SHARDS_COUNT);
            SearchMapShard shard = searchMapRow.get(shardId);
            shardedClouds.computeIfAbsent(shard, (k) -> new LinkedHashSet<>()).add(cloudId);
        }

        for (Map.Entry<SearchMapShard, Set<String>> entry: shardedClouds.entrySet()) {
            rule.execute(
                new BasicYcSearchContext(proxy, multiCloudContext, entry.getValue()),
                mfcb.newCallback());
        }

        mfcb.done();
    }

    private static class ConcatCallback
        extends AbstractFilterFutureCallback<List<YcSearchResult<BasicYcResultItem>>, YcSearchResult<BasicYcResultItem>>
    {
        private final MultiCloudYCSearchContext context;

        public ConcatCallback(
            final FutureCallback<? super YcSearchResult<BasicYcResultItem>> callback,
            final MultiCloudYCSearchContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final List<YcSearchResult<BasicYcResultItem>> results) {
            YcSearchResult<BasicYcResultItem> result =
                new YcSearchResult<>(results.size() * context.length());
            for (YcSearchResult<BasicYcResultItem> item: results) {
                result.addAll(item);
            }

            callback.completed(result);
        }
    }
}
