package ru.yandex.search.yc;

import java.io.IOException;
import java.util.function.Supplier;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class YcMalformedDocsListHandler implements ProxyRequestHandler {
    private static final User USER = new User(YcConstants.YC_QUEUE, new StringPrefix("0"));
    private final YcSearchProxy server;

    public YcMalformedDocsListHandler(final YcSearchProxy server) {
        this.server = server;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        MalformedDocsContext context = new MalformedDocsContext(session, server);

        QueryConstructor qc = new QueryConstructor("/search-malformed?");
        qc.append("prefix", "0");
        qc.append("db", "malformed_search_docs");
        qc.append("get", "*");

        if (context.offset() > 0) {
            qc.append("offset", context.offset());
        }

        qc.append("sort", "yc_mf_message_transfer_ts");

        if (context.service() != null) {
            qc.append("text", "yc_mf_service:" + context.service);
            if (context.length() > 0) {
                qc.append("length", context.length());
            } else {
                qc.append("length", "1");
            }
        } else {
            qc.append("text", "id:*");
            qc.append("group", "yc_mf_service");
            qc.append("merge_func", "none");
            if (context.length() > 0) {
                qc.append("length", context.length());
            }
        }
        ///search?prefix=0&db=malformed_search_docs&text=id:*&hr&get=*&length=10&group=yc_mf_service&merge_func=none&sort=yc_mf_message_transfer_ts'
        server.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            context.failoverDelay(),
            context.localityShuffle(),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Printer(context));
    }

    private static class Printer extends AbstractProxySessionCallback<JsonObject> {
        private final MalformedDocsContext context;

        public Printer(final MalformedDocsContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final JsonObject searchResultObj) {
            StringBuilderWriter sbw = new StringBuilderWriter();

            try (JsonWriter writer = context.jsonType().create(sbw)) {
                JsonMap searchResult = searchResultObj.asMap();
                int hitsCount = searchResult.getInt("hitsCount");
                JsonList hitsArray = searchResult.getList("hitsArray");
                int length;
                if (context.length() <= 0) {
                    length = Integer.MAX_VALUE;
                } else {
                    length = context.offset() + context.length();
                }
                boolean hasNext = context.offset() + hitsArray.size() >= hitsCount;
                context.session().connection().setSessionInfo(
                    SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
                    Long.toString(hitsArray.size()));

                writer.startObject();
                writer.key("has_next");
                writer.value(hasNext);
                writer.key("total");
                writer.value(hitsCount);
                writer.key("documents");
                writer.startArray();
                for (int i = context.offset; i < Math.min(hitsArray.size(), length); i++) {
                    writer.value(hitsArray.get(i));
                }
                writer.endArray();
                writer.endObject();
            } catch (JsonException | IOException e) {
                failed(e);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
        }
    }

    private static class MalformedDocsContext
        extends TokenRequiredYcSearchContext
        implements UniversalSearchProxyRequestContext
    {
        protected final ProxySession session;
        protected final JsonType jsonType;
        protected final AsyncClient client;
        protected final YcSearchProxy proxy;
        protected final boolean allowLaggingHosts;
        protected final boolean debug;
        protected final int length;
        protected final int offset;
        protected final String service;
        protected final Supplier<? extends HttpClientContext> contextGenerator;

        public MalformedDocsContext(
            final ProxySession session,
            final YcSearchProxy proxy)
            throws BadRequestException
        {
            super(session);
            this.session = session;
            this.proxy = proxy;

            CgiParams params = session.params();
            jsonType = JsonTypeExtractor.NORMAL.extract(params);
            allowLaggingHosts =
                params.getBoolean("allow-lagging-hosts", true);

            client = proxy.searchClient().adjust(session.context());
            debug = params.getBoolean("debug", false);
            length = params.get("length", 0, PositiveIntegerValidator.INSTANCE);
            offset = params.get("offset", 0, NonNegativeIntegerValidator.INSTANCE);
            service = params.getString("service", null);
            contextGenerator =
                session.listener().createContextGeneratorFor(client);
        }

        public long failoverDelay() {
            return 1000;
        }

        public boolean localityShuffle() {
            return false;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public YcSearchProxy proxy() {
            return proxy;
        }

        public boolean debug() {
            return debug;
        }

        public int length() {
            return length;
        }

        public int offset() {
            return offset;
        }

        public String service() {
            return service;
        }

        @Override
        public User user() {
            return USER;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public PrefixedLogger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return allowLaggingHosts ? Long.MAX_VALUE : 0L;
        }

        public Supplier<? extends HttpClientContext> contextGenerator() {
            return contextGenerator;
        }
    }

}
