package ru.yandex.search.yc;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.field.PrefixedStoredField;
import ru.yandex.search.request.util.BoostByOrderFieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.search.yc.iam.AuthorizationResolution;
import ru.yandex.search.yc.iam.ResourceWithResolution;
import ru.yandex.yc.search.YcSearchFields;

public class YcPlainSearchRule<T extends RequestProvider & YcContextProvider>
    implements SearchRule<T, YcSearchResult<BasicYcResultItem>>
{
    private static final JsonObject EMPTY_RESPONSE;
    static {
        JsonMap empty = new JsonMap(BasicContainerFactory.INSTANCE);
        empty.put("hitsArray", new JsonList(BasicContainerFactory.INSTANCE));
        empty.put("hitsCount", new JsonLong(0L));
        EMPTY_RESPONSE = empty;
    }

    private static final List<String> ATT_SEARCH_FIELDS =
        Collections.unmodifiableList(
            Arrays.asList(
                YcFields.ATTRIBUTE_VALUE.prefixedField()));
    public static final List<PrefixedStoredField> MAIN_DOC_SEARCH_FIELDS =
        Collections.unmodifiableList(
            Arrays.asList(
                YcSearchFields.YC_RESOURCE_ID,
                YcSearchFields.YC_NAME));

    public static final List<String> MAIN_DOC_SEARCH_FIELDS_STR =
        Collections.unmodifiableList(
            MAIN_DOC_SEARCH_FIELDS.stream().map(PrefixedStoredField::prefixed).collect(Collectors.toList()));
    public static final BoostByOrderFieldsTermsSupplierFactory MAIN_DOC_TERMS_SUPPLIER =
        new BoostByOrderFieldsTermsSupplierFactory(2f, 2f, MAIN_DOC_SEARCH_FIELDS_STR);
    public static final BoostByOrderFieldsTermsSupplierFactory ATT_DOC_TERMS_SUPPLIER =
        new BoostByOrderFieldsTermsSupplierFactory(1f, 1f, ATT_SEARCH_FIELDS);
    public static final BoostByOrderFieldsTermsSupplierFactory ALL_DOC_TERMS_SUPPLIER =
        new BoostByOrderFieldsTermsSupplierFactory(
            2f,
            1f,
            Collections.unmodifiableList(
                Arrays.asList(
                    YcSearchFields.YC_NAME.prefixed(),
                    YcSearchFields.YC_ATTVALUE.prefixed(),
                    YcSearchFields.YC_RESOURCE_ID.prefixed())));

    public static final String LEFT_JOIN_DP = "get_main_doc";
//        "left_join(" + YcFields.MAIN_DOC_ID.field()
//            + ',' + YcFields.ID.field()
//            // get/out fields
//            + ",," + YcFields.ATTRIBUTES.field() + ',' + YcSearchFields.YC_RESOURCE_PATH.stored() + ')';
    private static final String DOC_ID_DP =
        "fallback(" + YcFields.MAIN_DOC_ID.storeField() + ',' + YcFields.ID.storeField() + " doc_id)";

    private final Boolean exactOnly;

    public YcPlainSearchRule() {
        this.exactOnly = false;
    }

    public YcPlainSearchRule(final boolean exactOnly) {
        this.exactOnly = exactOnly;
    }

    private void addPrioritizedByFolderQuery(
        SearchRequestText requestTextExact, SearchRequestText requestTextSuggest, StringBuilder query, String folderId,
        boolean exactOnly)
    {
        StringBuilder exactQueryText = new StringBuilder();
        requestTextExact.fieldsQuery(exactQueryText, YcPlainSearchRule.ALL_DOC_TERMS_SUPPLIER);

        if (!exactOnly) {
            StringBuilder suggestQueryText = new StringBuilder();
            requestTextSuggest.fieldsQuery(suggestQueryText, YcPlainSearchRule.ALL_DOC_TERMS_SUPPLIER);

            query.append('(');
            query.append(exactQueryText);
            query.append(" AND (");
            query.append(YcFields.FOLDER_ID.prefixedField());
            query.append(':');
            query.append(folderId);
            query.append(')');
            query.append(")^10");

            query.append(" OR (");
            query.append(exactQueryText);
            query.append(")^5");

            query.append(" OR (");
            query.append(suggestQueryText);
            query.append(" AND (");
            query.append(YcFields.FOLDER_ID.prefixedField());
            query.append(':');
            query.append(folderId);
            query.append(')');
            query.append(")^2");

            query.append(" OR (");
            query.append(suggestQueryText);
            query.append(")^1");
        } else {
            query.append('(');
            query.append(exactQueryText);
            query.append(" AND (");
            query.append(YcFields.FOLDER_ID.prefixedField());
            query.append(':');
            query.append(folderId);
            query.append(')');
            query.append(")^2");

            query.append(" OR (");
            query.append(exactQueryText);
            query.append(")^1");
        }
    }

    @Override
    public void execute(
        final T input, final FutureCallback<? super YcSearchResult<BasicYcResultItem>> callback)
        throws HttpException
    {
        String request = input.request().toLowerCase(Locale.ROOT);
        BasicYcSearchContext context = input.searchContext();

        StringBuilder attsQueryText = new StringBuilder();

        SearchRequestText requestTextExact = SearchRequestText.parse(
            request,
            x -> x,
            y -> y);

        SearchRequestText requestTextSuggest = SearchRequestText.parseSuggest(
            request,
            context.locale());

        String folderId = context.currentFolderId();
        if (requestTextExact.hasWords()) {
            if (folderId != null) {
                addPrioritizedByFolderQuery(requestTextExact, requestTextSuggest, attsQueryText, folderId, exactOnly);
            } else if (exactOnly) {
                requestTextExact.fieldsQuery(attsQueryText, ALL_DOC_TERMS_SUPPLIER);
            } else {
                requestTextSuggest.fieldsQuery(attsQueryText, ALL_DOC_TERMS_SUPPLIER);
            }
        } else {
            attsQueryText.append(YcFields.DOC_TYPE.prefixedField());
            attsQueryText.append(':');
            attsQueryText.append(YcDocType.FULL_DOC.fieldValue());
            attsQueryText.append(" AND ");
            attsQueryText.append(YcFields.RECORD_ID.prefixedField());
            attsQueryText.append(":*");
        }

        attsQueryText.append(" AND NOT yc_deleted_p:1 AND NOT yc_stale_p:1");

        SearchCallback<T> searchCallback =
            new SearchCallback<>(input, context, exactOnly, callback);

        QueryConstructor query =
            new QueryConstructor(
                "/search-yc?IO_PRIO=0&json-type=dollar");

        query.append("service", context.user().service());
        query.append("scorer", "lucene");
        query.append("sort", "#score");
        query.append("get", "*,#score");
        //query.append("debug", "true");
        query.append("dp", LEFT_JOIN_DP);
        query.append("group", YcFields.MAIN_DOC_ID.field());
        query.append("length", context.length() * 3);
        for (String cloudId: context.cloudIds()) {
            query.append("prefix", cloudId);
        }

        query.append("lowercase-expanded-terms", "true");
        query.append("replace-ee-expanded-terms", "true");
        //query.append("text", "yc_attvalue:*");
        //int curSbLength = query.sb().length();
        query.append("text", attsQueryText.toString());
        context.logger().info(
            "Hosts for user " + context.user + " " + context.proxy.searchMap().hosts(context.user));
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.proxy().config().searchFailOverDelay(),
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            searchCallback);
    }

    private static final class SearchCallback<T extends RequestProvider & YcContextProvider>
        implements FutureCallback<JsonObject>
    {
        private final T input;
        private final BasicYcSearchContext context;
        private final boolean exactOnly;
        private final long fetchTime;
        private final FutureCallback<? super YcSearchResult<BasicYcResultItem>> callback;

        private SearchCallback(
            final T input,
            final BasicYcSearchContext context,
            final boolean exactOnly,
            final FutureCallback<? super YcSearchResult<BasicYcResultItem>> callback)
        {
            this.input = input;
            this.context = context;
            this.exactOnly = exactOnly;
            this.fetchTime = System.currentTimeMillis();
            this.callback = callback;
        }

        @Override
        public void failed(final Exception ex) {
            callback.failed(ex);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        private void complete(
            final JsonMap map,
            final List<YcSearchResultItem> result)
            throws JsonException
        {
            JsonList hits = map.getList("hitsArray");

            if (context.debug()) {
                context.logger().info(
                    "lucene response "
                        + JsonType.HUMAN_READABLE.toString(hits));
            }

            for (JsonObject jo: hits) {
                JsonMap item = jo.asMap();
                YcSearchResultItem resultItem = new YcSearchResultItem(item, context);
                if (!context.cloudIds().contains(resultItem.cloudId())) {
                    context.logger().warning(
                        "Wrong cloud id in response "
                            + JsonType.NORMAL.toString(resultItem)
                            + " expected " + context.cloudIds());
                    continue;
                }
                if (!context.proxy().config().allowedServices().test(
                        resultItem.service()))
                {
                    context.logger().warning(
                        "Service in response not in allowed list "
                            + JsonType.NORMAL.toString(resultItem)
                            + " allowed " + context.proxy().config().allowedServices());

                    continue;
                }
                result.add(resultItem);

                //context.logger().info("Got resource " + item.getString("id") + " " + item.getInt("#score"));
            }
        }

        @Override
        public void completed(final JsonObject response) {
            context.logger().info(
                "Search finished in "
                    + (System.currentTimeMillis() - fetchTime) + "ms");

            YcSearchResult<YcSearchResultItem> result = new YcSearchResult<>(context.length());
            try {
                complete(response.asMap(), result);
                context.logger().info("Result size, before filter " + result.size());
                if (result.size() < context.length() && exactOnly) {
                    try {
                        context.logger().info("Not enough exact matches, making a suggest request...");
                        new YcPlainSearchRule<>(false).execute(input, callback);
                    } catch (HttpException e) {
                        callback.failed(e);
                    }
                } else {
                    MultiFutureCallback<ResourceWithResolution<YcSearchResultItem>> mfcb =
                        new MultiFutureCallback<>(
                            new FilterByFolderCallback<>(
                                context,
                                result.size(),
                                callback));
                    for (YcSearchResultItem item : result) {
                        context.proxy().resourceFilter().authorizeFolder(
                            context.logger(),
                            context.iamToken(),
                            item,
                            mfcb.newCallback());
                    }
                    mfcb.done();
                }
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private static class FilterByFolderCallback<T extends RequestProvider & YcContextProvider>
        extends AbstractFilterFutureCallback<List<ResourceWithResolution<YcSearchResultItem>>, YcSearchResult<BasicYcResultItem>>
    {
        private final BasicYcSearchContext context;
        private final int resources;

        public FilterByFolderCallback(
            final BasicYcSearchContext context,
            final int resources,
            final FutureCallback<? super YcSearchResult<BasicYcResultItem>> callback)
        {
            super(callback);

            this.resources = resources;
            this.context = context;
        }

        @Override
        public void completed(
            final List<ResourceWithResolution<YcSearchResultItem>> resourcesWithResolution)
        {
            int resolutionDrop = 0;
            YcSearchResult<BasicYcResultItem> keepResources =
                new YcSearchResult<>(resourcesWithResolution.size());

            for (ResourceWithResolution<YcSearchResultItem> res: resourcesWithResolution) {
                if (res.resolution() == AuthorizationResolution.ALLOW) {
                    try {
                        res.resource().highlights(false, false, true);
                    } catch (JsonException je) {
                        failed(je);
                    }
                    keepResources.add(res.resource());
                } else {
                    resolutionDrop += 1;
                }
            }

            int totalDropped = resources - keepResources.size();
            if (totalDropped > 0) {
                StringBuilder log = new StringBuilder();
                log.append("Iam before filter: ");
                log.append(resources);
                log.append(" dropped by iam: ");
                log.append(resolutionDrop);
                log.append(" other drops: ");
                log.append(resources - resolutionDrop - keepResources.size());
                context.logger().warning(log.toString());
            }
            callback.completed(keepResources);
        }
    }

    private static class AfterFiltrationCallback
        extends AbstractFilterFutureCallback<List<YcSearchResultItem>, YcSearchResult<BasicYcResultItem>>
    {
        public AfterFiltrationCallback(
            final FutureCallback<? super YcSearchResult<BasicYcResultItem>> callback)
        {
            super(callback);
        }

        @Override
        public void completed(final List<YcSearchResultItem> items) {
            YcSearchResult<BasicYcResultItem> result = new YcSearchResult<>(items);
            // here we should check if there are more records in index
            // to fill losses made by iam
            callback.completed(result);
        }
    }
}


