package ru.yandex.search.yc;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.yc.iam.AuthorizationResolution;
import ru.yandex.search.yc.iam.ResourceWithResolution;

public class YcSearchResourceMetaHandler implements ProxyRequestHandler {
    private final YcSearchProxy proxy;

    public YcSearchResourceMetaHandler(final YcSearchProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        MetaResourceRequestContext context;
        if (session.request() instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) session.request()).getEntity();
            context = new MetaResourceRequestContext(session, entity);
        } else {
            context = new MetaResourceRequestContext(session, null);
        }

        QueryConstructor qc = new QueryConstructor("/search-resource-meta?");
        qc.append("prefix", context.user.prefix().toStringFast());
        qc.append("service", context.user.service());
        qc.append("get", "*");
        StringBuilder sb = new StringBuilder();
        sb.append("yc_doc_type_p:main AND yc_resource_id:(");
        for (String resource: context.resources()) {
            sb.append(resource);
            sb.append(' ');
        }
        sb.setLength(sb.length() - 1);
        sb.append(')');
        if (!context.returnDeleted()) {
            sb.append("AND NOT yc_deleted:1");
        }
        qc.append("text", sb.toString());

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            context.failoverDelay(),
            context.localityShuffle(),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.client().httpClientContextGenerator(),
            new Callback(context));
    }

    private static class ResourcesPrinter
        extends AbstractProxySessionCallback<List<ResourceWithResolution<BasicYcResultItem>>>
    {
        private final MetaResourceRequestContext context;

        public ResourcesPrinter(final MetaResourceRequestContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(
            final List<ResourceWithResolution<BasicYcResultItem>> filtered)
        {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.jsonType().create(sbw)){
                writer.startObject();
                writer.key("resources");
                writer.startArray();

                for (ResourceWithResolution<BasicYcResultItem> item: filtered) {
                    if (item.resolution() == AuthorizationResolution.ALLOW) {
                        writer.value(item.resource());
                    } else {
                        context.logger().warning("Dropping by iam " + item.resource().docId());
                    }

                }
                writer.endArray();
                writer.endObject();
            } catch (IOException e) {
                failed(e);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
        }
    }

    private class Callback extends AbstractProxySessionCallback<JsonObject> {
        private final MetaResourceRequestContext context;

        public Callback(final MetaResourceRequestContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final JsonObject backendResultObj) {
            List<BasicYcResultItem> result;
            try {
                JsonList list = backendResultObj.asMap().getList("hitsArray");
                result = new ArrayList<>(list.size());
                for (JsonObject item: list) {
                    result.add(new BasicYcResultItem(item.asMap()));
                }

            } catch (JsonException e) {
                failed(e);
                return;
            }

            proxy.resourceFilter().authorize(
                context.logger(),
                context.iamToken(),
                result,
                new ResourcesPrinter(context));
        }
    }

    private class MetaResourceRequestContext
        extends TokenRequiredYcSearchContext
        implements UniversalSearchProxyRequestContext
    {
        private final String cloudId;
        private final Set<String> resources;
        private final ProxySession session;
        private final User user;
        private final AsyncClient client;
        private final JsonType jsonType;
        private final Long failoverDelay;
        private final boolean localityShuffle;
        private final boolean returnDeleted;

        public MetaResourceRequestContext(
            final ProxySession session,
            final HttpEntity entity)
            throws HttpException, IOException
        {
            super(session);

            this.session = session;
            this.cloudId = session.params().getString("cloud_id");
            if (entity != null) {
                try {
                    JsonMap root = TypesafeValueContentHandler.parse(CharsetUtils.content(entity)).asMap();
                    JsonList resIdsJsonList = root.getList("res_ids");
                    Set<String> resIds = new LinkedHashSet<>(resIdsJsonList.size() << 1);
                    for (JsonObject resIdObj: resIdsJsonList) {
                        resIds.add(resIdObj.asString());
                    }
                    this.resources = resIds;
                } catch (JsonException je) {
                    throw new BadRequestException("Failed to parse body", je);
                }
            } else {
                this.resources =
                    session.params().get(
                        "res_ids",
                        new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new));
            }
            this.user = new User(YcConstants.YC_QUEUE, new StringPrefix(cloudId));
            this.client = proxy.searchClient().adjust(session.context());
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
            this.failoverDelay = session.params().getLong("failover-delay", 100L);
            this.localityShuffle = session.params().getBoolean("locality-shuffle", true);
            this.returnDeleted = session.params().getBoolean("return-deleted", false);
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public boolean returnDeleted() {
            return returnDeleted;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public PrefixedLogger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return 0L;
        }

        public String cloudId() {
            return cloudId;
        }

        public Set<String> resources() {
            return resources;
        }

        public ProxySession session() {
            return session;
        }

        public Long failoverDelay() {
            return failoverDelay;
        }

        public boolean localityShuffle() {
            return localityShuffle;
        }
    }

}
